#include <starmicronics/IOPortConnection.h>

// forward declarations
BOOL PrintStart(UINT32 file_no);
BOOL SendDataToPrinter(void * port, UCHAR const * writeBuffer, UINT32 writeBufferSize);
BOOL GetOnlineStatusUsage(void * port);
BOOL GetParsedStatusUsage(void * port);
BOOL BeginCheckedBlockUsage(void * port);
BOOL EndCheckedBlockUsage(void * port);
BOOL WritePortUsage(void * port, UCHAR const * writeBuffer, UINT32 writeBufferSize);
UINT32 ReadPrintDataFile(PBYTE * printData, UINT32 file_no);

#if defined(WIN32) || defined(_WIN32_WCE)

#include <windows.h>
#include <stdio.h>
#include <wchar.h>

// unicode
#define CONSOLE_OUT wprintf
#define CONSOLE_IN wscanf
#define CONSOLE_IN_CHAR getwchar

#define STR_CMP wcscmp
#define STR_TO_INT _wtoi

#define STR_ELEM wchar_t
#define STR(x) L ## x

#define ESC_POS_B L"BSC10_ESCPOS_Bitimage.prn"
#define ESC_POS_T L"BSC10_ESCPOS_Text.prn"

wchar_t g_portName[100 + 1] = {0};
wchar_t g_portSettings[100 + 1] = {0};
const UINT32 g_portTimeout = 5 * 1000;

int wmain(int argc, wchar_t *argv[ ], wchar_t *envp[ ])

#else

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>

// ansi
#define CONSOLE_OUT printf
#define CONSOLE_IN scanf
#define CONSOLE_IN_CHAR getchar

#define STR_CMP strcmp
#define STR_TO_INT atoi

#define STR_ELEM char
#define STR(x) x

#define ESC_POS_B "BSC10_ESC-POS_Bitimage.prn"
#define ESC_POS_T "BSC10_ESC-POS_Text.prn"

char g_portName[100 + 1] = {0};
char g_portSettings[100 + 1] = {0};
const UINT32 g_portTimeout = 10 * 1000;

int main(int argc, char * argv[])

#endif
{
    CONSOLE_OUT(STR("Enter the port name: "));
    CONSOLE_IN(STR("%[^\t\n]"), g_portName); CONSOLE_IN_CHAR();

    CONSOLE_OUT(STR("Enter the port settings: "));
    CONSOLE_IN(STR("%[^\t\n]"), g_portSettings); CONSOLE_IN_CHAR();

    CONSOLE_OUT(STR("\n"));

    while(true)
    {
        CONSOLE_OUT(STR("[1] ESC/POS Bitmap print\n"));
        CONSOLE_OUT(STR("[2] ESC/POS Text   print\n"));
        CONSOLE_OUT(STR("[x] Quit\n"));

        STR_ELEM answerString[10];

        CONSOLE_OUT(STR("Select one of the above options: "));
        CONSOLE_IN(STR("%s"), answerString);
        CONSOLE_IN_CHAR();

        if(STR_CMP(answerString, STR("x")) == 0)
        {
            return 0;
        }

        int answer = STR_TO_INT(answerString);

        switch(answer)
        {
        case 1:
        case 2:
            PrintStart(answer);
            break;
        default:
            CONSOLE_OUT(STR("Invalid Option\n"));
            break;
        }

        CONSOLE_OUT(STR("\n"));
    }

    return 0;
}

BOOL PrintStart(UINT32 file_no)
{
    void * port = OpenPort(g_portName, g_portSettings, g_portTimeout);

    if(port == NULL)
    {
        CONSOLE_OUT(STR("Failure - OpenPort\n"));
        return FALSE;
    }

    if (GetParsedStatusUsage(port) == FALSE)
    {
        CONSOLE_OUT(STR("The pritner is offline.\n"));

        ClosePort(port);
        return FALSE;
    }

    BYTE * rawData;
    UINT32 ReadFileSize = ReadPrintDataFile(&rawData, file_no);

    if (ReadFileSize == 0)
    {
        CONSOLE_OUT(STR("Error - no print data.\n"));

        ClosePort(port);
        return FALSE;
    }

    if (SendDataToPrinter(port, rawData, ReadFileSize))
    {
        CONSOLE_OUT(STR("Success!!\n"));
    }
    else
    {
        CONSOLE_OUT(STR("Failure!!\n"));
    }

    free(rawData);

    ClosePort(port);

    return TRUE;
}

BOOL SendDataToPrinter(void * port, UCHAR const * writeBuffer, UINT32 writeBufferSize)
{
    BOOL result = TRUE;

    do
    {
        if (BeginCheckedBlockUsage(port) == FALSE)
        {
            result = FALSE;
            break;
        }

        if (WritePortUsage(port, writeBuffer, writeBufferSize) == FALSE)
        {
            GetParsedStatusUsage(port);

            result = FALSE;
            break;
        }

        if (EndCheckedBlockUsage(port) == FALSE)
        {
            result = FALSE;
            break;
        }

    } while(0);


    if (result == FALSE)
    {
        STR_ELEM retryPrinting[10];

        while(true)
        {
            CONSOLE_OUT(STR("Retry printing? (y/n) : "));
            CONSOLE_IN(STR("%s"), retryPrinting);
            CONSOLE_IN_CHAR();

            if(STR_CMP(retryPrinting, STR("y")) == 0)
            {
                break;
            }
            else if(STR_CMP(retryPrinting, STR("n")) == 0)
            {
                return FALSE;
            }

            CONSOLE_OUT(STR("\n"));
        }

        CONSOLE_OUT(STR("The printer online and hit enter"));
        CONSOLE_IN_CHAR();

        return SendDataToPrinter(port, writeBuffer, writeBufferSize);
    }

    return result;
}

BOOL GetOnlineStatusUsage(void * port)
{
    SM_BOOLEAN onlineStatus;

    if(GetOnlineStatus(port, &onlineStatus) == SM_FALSE)
    {
        CONSOLE_OUT(STR("Failure - GetOnlineStatus\n"));
        return FALSE;
    }

    if(onlineStatus == SM_TRUE)
    {
        return TRUE;    // ONLINE
    }

    return FALSE;
}

BOOL GetParsedStatusUsage(void * port)
{
    StarPrinterStatus_0 status;

    if (GetParsedStatusEx(port, &status, 0) == SM_FALSE)
    {
        CONSOLE_OUT(STR("Failure - GetParsedStatusEx\n"));
        return FALSE;
    }

    if (status.offline == SM_TRUE)
    {
        if(status.coverOpen == SM_TRUE)
        {
            CONSOLE_OUT(STR("The printer is offline - cover open.\n"));
        }

        if(status.paperEmpty == SM_TRUE)
        {
            CONSOLE_OUT(STR("The printer is offline - paper empty.\n"));
        }

        return FALSE;
    }

    return TRUE;        // ONLINE
}

BOOL BeginCheckedBlockUsage(void * port)
{
    StarPrinterStatus_0 status;

    if (BeginCheckedBlockEx(port, &status, 0) == SM_FALSE)
    {
        CONSOLE_OUT(STR("Failure - BeginCheckedBlockEx\n"));
        return  FALSE;
    }

    if(status.offline == SM_TRUE)
    {
        CONSOLE_OUT(STR("Error - Cannot proceed with printing because the printer is offline.\n"));

        if(status.coverOpen == SM_TRUE)
        {
            CONSOLE_OUT(STR("The printer is offline - cover open.\n"));
        }
        if(status.paperEmpty == SM_TRUE)
        {
            CONSOLE_OUT(STR("The printer is offline - paper empty.\n"));
        }

        return  FALSE;
    }

    return TRUE;
}

BOOL EndCheckedBlockUsage(void * port)
{
    StarPrinterStatus_0 status;

    if (EndCheckedBlockEx(port, &status, 0) == SM_FALSE)
    {
        CONSOLE_OUT(STR("Failure - EndCheckedBlockEx\n"));
        return  FALSE;
    }

    if(status.offline == SM_TRUE)
    {
        CONSOLE_OUT(STR("Error - The printer is offline indicating that the printing failed.\n"));

        if(status.coverOpen == SM_TRUE)
        {
            CONSOLE_OUT(STR("The printer is offline - cover open.\n"));
        }
        if(status.paperEmpty == SM_TRUE)
        {
            CONSOLE_OUT(STR("The printer is offline - paper empty.\n"));
        }

        return  FALSE;
    }

    return TRUE;
}

BOOL WritePortUsage(void * port, UCHAR const * writeBuffer, UINT32 writeBufferSize)
{
    UINT32 zeroProgressOccurances = 0;
    UINT32 totalSizeCommunicated  = 0;

    while ((totalSizeCommunicated < writeBufferSize) && (zeroProgressOccurances < 2)) // adjust zeroProgressOccurances threshold as required
    {
        UINT32 sizeCommunicated = 0;
        if(WritePort(port, &writeBuffer[totalSizeCommunicated], writeBufferSize - totalSizeCommunicated, &sizeCommunicated) == 0)
        {
            totalSizeCommunicated = -1;
            break;
        }

        if (sizeCommunicated == 0)
        {
            zeroProgressOccurances++;
        }
        else
        {
            totalSizeCommunicated += sizeCommunicated;
            zeroProgressOccurances = 0;
        }
    }

    if (totalSizeCommunicated != writeBufferSize)
    {
        if (totalSizeCommunicated == -1)
        {
            CONSOLE_OUT(STR("Failure - WritePort.\n"));
        }
        else
        {
            CONSOLE_OUT(STR("Error - not all data could be written out.\n"));
        }

        return FALSE;
    }

    return TRUE;
}

UINT32 ReadPrintDataFile(PBYTE * printData, UINT32 file_no)
{
    HANDLE fileHandle;
    DWORD  fileSize;
    DWORD  readSize;
    PBYTE  pData         = NULL;
    UINT32 printDataSize = 0;

    /* Read print data from file */
    switch(file_no)
    {
    case 1:
        fileHandle = CreateFile(ESC_POS_B, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
        break;
    case 2:
        fileHandle = CreateFile(ESC_POS_T, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
        break;
    }
    
    if (fileHandle == INVALID_HANDLE_VALUE)
    {
        return 0;
    }

    fileSize = GetFileSize(fileHandle, &fileSize);
    if (fileSize == INVALID_FILE_SIZE)
    {
        CloseHandle(fileHandle);
        return 0;
    }

    printDataSize = fileSize;
    pData = (PBYTE)malloc(printDataSize);
    if (pData == NULL)
    {
        CloseHandle(fileHandle);
        return 0;
    }

    if (ReadFile(fileHandle, pData, printDataSize, &readSize, NULL) == FALSE)
    {
        free(pData);
        CloseHandle(fileHandle);
        return 0;
    }

    CloseHandle(fileHandle);

    *printData = pData;

    return printDataSize;
}
