#include "stdafx.h"
#include "utils.h"
#include "serial.h"
#include "selector.h"

#include "commands.h"
#include "options.h"
#include "functions.h"
#include "extensions.h"
#include "structsIn.h"
#include "structsOut.h"

#include <direct.h>
#include <stdlib.h>
#include <errno.h>


static unsigned char uchSequenceNumber = MIN_SEQUENCE_NUMBER;

void vSleep( clock_t wait )
{
   clock_t goal;
   goal = wait + clock();
   while( goal > clock() );
}

bool bIs_Hexadecimal( unsigned char uchByte)
{
	if( ( uchByte < 0x30 || uchByte > 0x39 ) && ( tolower(uchByte) < 'a' || tolower(uchByte) > 'f' ) )
		return false;

	return true;
}

bool bIs_Escaped_Char( unsigned char uchByte )
{
	if( uchByte == 0x02 || uchByte == 0x03 || uchByte == 0x1A ||
		uchByte == 0x1B || uchByte == 0x1C || uchByte == 0x1D ||
		uchByte == 0x1E || uchByte == 0x1F                       )
	{
		return true;
	}
	else
	{
		return false;
	}
}

unsigned short sHexadecimal_Value( unsigned char uchHighNibble, unsigned char uchLowNibble )
{
	if( uchHighNibble >= 0x30 && uchHighNibble <= 0x39 )
		uchHighNibble -= 0x30;
	else
		uchHighNibble = tolower(uchHighNibble) - 'a' + 0x0a;

	if( uchLowNibble >= 0x30 && uchLowNibble <= 0x39 )
		uchLowNibble -= 0x30;
	else
		uchLowNibble = tolower( uchLowNibble ) - 'a' + 0x0a;

	return ( (uchHighNibble) << 4) + uchLowNibble;
}

void vSetSequenceNumber( void )
{
	time_t tTime;

	srand( (unsigned int) time( &tTime ) );

   unsigned char tempseq = (unsigned char) ( rand() % (MAX_SEQUENCE_NUMBER - MIN_SEQUENCE_NUMBER + 1) + 
		                                  MIN_SEQUENCE_NUMBER );
   if ( tempseq == ( uchSequenceNumber - 1 ) )
	   uchSequenceNumber +=2;
   else
      uchSequenceNumber = tempseq;
}

unsigned char uchGetSequenceNumber( void )
{
	uchSequenceNumber += 0x01;

	if( uchSequenceNumber < MIN_SEQUENCE_NUMBER ||
		uchSequenceNumber > MAX_SEQUENCE_NUMBER )
	{
		uchSequenceNumber = MIN_SEQUENCE_NUMBER;
	}

	return uchSequenceNumber;
}

short sTenPow( int iPow )
{
	int iCount;
	short sAnswer = 1;

	if( iPow == 0 )
		return sAnswer;

	if( iPow > 0 )
		for( iCount = 0; iCount < iPow; iCount++ )
			sAnswer *= 10;
	else
		for( iCount = iPow; iCount < 0; iCount++ )
			sAnswer /= 10;

	return sAnswer;
}

int iField_Numeric_Value( unsigned char * szField,  int iLength, int *piResponse )
{
	int iTotalValue;
	int iCounter;

	iTotalValue = 0;
	*piResponse = 0;

	for( iCounter = iLength; iCounter > 0; iCounter-- )
	{
		if( szField[iLength - iCounter] < 0x30 || szField[iLength - iCounter] > 0x39 )
			return ERROR_SERIAL_INVALID_FIELD_TYPE;

		iTotalValue += (szField[iLength - iCounter] - 0x30) *  sTenPow(iCounter - 1 );
	}

	*piResponse = iTotalValue;

	return SERIAL_NO_ERROR;
}

/*-----------------------------------------------------------------------------
   Configuration
-----------------------------------------------------------------------------*/
#define BASE_YEAR 2000
#define MAX_YEAR 2099

/*bad julian day*/
#define BAD_DATE 36600U
#define BAD_DAY 0
#define BAD_MONTH 0
#define BAD_YEAR 0

#define MAX_DATE 36524U           /*  max julian day (31/12/2099)   */

//#define FIRST_DAY_OF_WEEK day_saturday
#define BAD_TIME 86500UL
#define BAD_HOUR 24
#define BAD_MIN 60
#define BAD_SEC 60

#define HOURS_IN_DAY 24           /*  number of hours in a day      */
#define SECONDS_IN_DAY 86400UL    /*  number of seconds in a day    */
#define SECONDS_IN_HOUR 3600      /*  number of seconds in an hour  */
#define SECONDS_IN_MINUTE 60      /*  number of seconds in a minute */
#define MINUTES_IN_HOUR 60        /*  number of minutes in an hour  */

#define FIRST_TWO_MONTHS 59

#define NO_LEAP_DAYS 365          /*  number of days for no leap years */
#define LEAP_DAYS 366             /*  number of days for leap years */

#define JANUARY 1
#define FEBRUARY 2
#define MARCH 3
#define APRIL 4
#define MAY 5
#define JUNE 6
#define JULE 7
#define AUGUST 8
#define SEPTEMBER 9
#define OCTOBER 10
#define NOVEMBER 11
#define DECEMBER 12

#define DAYS_IN_WEEK 7U

/*-----------------------------------------------------------------------------
** Function Name : CAL_DateToDMY
** Description   : Convert from a julian date to month, day, year
** Input         : the date, day, month, year
** Output        : nothing
** Notes         : when the date is bad put BAD_DAY, BAD_MONTH and BAD_YEAR
-----------------------------------------------------------------------------*/
void CAL_DateToDMY( tDate date, word *day, word *month, word *year )
{
  dword calc = 0;

  if( date > MAX_DATE )
  {
    /* bad date */
    *day = BAD_DAY;
    *month = BAD_MONTH;
    *year = BAD_YEAR;
  }
  else
  {
    /* if the date is minor to 59, then the year is a Base Year */
    if( date <= FIRST_TWO_MONTHS )
    {
      *year = BASE_YEAR;

      /* Verify if the month is January or February */
      if( date <= 30 )
      {
        *month = JANUARY;
        *day = (word)( date + 1 );
      }
      else
      {
        *month = FEBRUARY;
        *day = (word)( date - 30 );
      }
    }
    else
    {
      calc = ( 4  * ( (dword)( date - FIRST_TWO_MONTHS ) ) - 1 );
      *year = (word)( calc / 1461 );

      calc = ( 5  * ( ( calc % 1461 ) / 4 ) ) + 2;

      *month = (word)( calc / 153 );
      *day = (word)( ( ( calc % 153 ) + 5 ) / 5 );

      if( *month < OCTOBER )
      {
        *month += 3;
      }
      else
      {
        *month -= 9;
        *year = *year + 1;
      }
      if( *year < 100 )
      {
        *year += BASE_YEAR;
      }
    }
  }
}

/*-----------------------------------------------------------------------------
** Function Name : CAL_TimeToHMS
** Description   : Converts a Time variable to Hours, Minutes, Seconds
** Input         : the time, hour, minute and second
** Output        : nothing
** Notes         : when the time is bad put BAD_HOUR, BAD_MIN and BAD_YEAR
-----------------------------------------------------------------------------*/
void CAL_TimeToHMS( tTime time, word *hour, word *min, word *sec )
{
  if( time < SECONDS_IN_DAY )
  {
    *hour = (word)( time / SECONDS_IN_HOUR );
    time -= ( (dword) *hour * SECONDS_IN_HOUR );
    *min = (word)( time / SECONDS_IN_MINUTE );
    time -= ( (dword)*min * SECONDS_IN_MINUTE );
    *sec = (word)time;
  }
  else
  {
    /* bad time */
    *hour = BAD_HOUR;
    *min = BAD_MIN;
    *sec = BAD_SEC;
  }
}

int iFloatToString( unsigned char * szString )
{
	int iContador;
	int iCommas = 0, iCommaPosition = 0;
	unsigned char szAux[17];

	for( iContador = 0; iContador < (int) strlen( (char *) szString ); iContador++ )
	{
		if( !isdigit(szString[iContador]) && szString[iContador] != '.' && szString[iContador] != ',')
			return ERROR;

		if( szString[iContador] == '.' || szString[iContador] == ',' )
		{
			iCommaPosition = iContador;
			iCommas++;
		}

		if( iCommas > 1 )
			return ERROR;
	}

	if( iCommas == 0 )
		return ERROR;

	if( iCommaPosition == 0 )
		sprintf( (char *) szAux, "%.14s00", szString );
	else
		sprintf( (char *) szAux, "%.*s%s", iCommaPosition , szString , &szString[ iCommaPosition + 1] );

	sprintf( (char *) szString, "%.0f", atof((char *) szAux) );

	return SUCCESS;

}


//Clear Left Zeros 
void vTrimZeroL ( CHAR * szTemp)
{
	INT iCount;
	INT iCStart;
	INT iC;

	iCount = (int)strlen(szTemp)-1 ;

	for (iCStart=0;(szTemp[iCStart]=='0' && iCStart<=iCount);iCStart++);

	if(iCStart > iCount)
	{
		sprintf(szTemp,"0");
	
	}
	else 
	{
		if(iCStart>0)
		{
			if(szTemp[iCStart] == ',')
				iCStart--;
				
			if(iCStart>0)
			{
				for (iC=0;(iC+iCStart)<=iCount;iC++)
					szTemp[iC]=szTemp[iC+iCStart];

				szTemp[iC]=0;
			}
		}
	}

}


void vCheckLogFlag(DWORD * dwLogAction)
{
	HKEY	m_Key;
	BYTE	*X;
	DWORD	dwLog,dwSize, dwType;

	if( RegOpenKeyEx( HKEY_CURRENT_USER, "SOFTWARE\\EPSON\\InterfaceEpson", 0x00, KEY_ALL_ACCESS, &m_Key ) == ERROR_SUCCESS )
	{
		dwSize=4;dwType=4;
		X = (BYTE *) &dwLog;
		if( RegQueryValueEx( m_Key, "Log", 0x00, &dwType , X, &dwSize ) == ERROR_SUCCESS)
		{
			*dwLogAction = dwLog;
			RegCloseKey( m_Key );
			return;
		}



	}

	*dwLogAction = 0;

	if( RegOpenKeyEx( HKEY_CURRENT_USER, "SOFTWARE\\EPSON\\InterfaceEpson", 0x00, KEY_ALL_ACCESS, &m_Key ) != ERROR_SUCCESS )
	{
			RegCreateKeyEx( HKEY_CURRENT_USER, "SOFTWARE\\EPSON\\InterfaceEpson", 0x00, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &m_Key, &dwLog );		
	}
		dwLog = 0;
		X = (BYTE *) &dwLog;
		RegSetValueEx( m_Key, "Log", 0x00, REG_DWORD, X, 0x04 );
		RegCloseKey( m_Key );


	return;

}


DWORD dwEpsonLogPath( LPSTR pszPath )
{
	if((isalpha(pszPath[0]) == 0) || (pszPath[1] != ':') && (pszPath[2] != '\\'))
	{
		iGetEpsonPath(pszPath);
		return SUCCESS;
	}

	if(_mkdir(pszPath) == 0)
		return SUCCESS;
	else
	{
		if(errno == EEXIST)
			return SUCCESS;
		else if (errno != ENOENT)
		{
			iGetEpsonPath(pszPath);
			return SUCCESS;
		}
	}

	return SUCCESS;
}

bool bStringToFloat( unsigned char *szString, unsigned char uchDecimalSeparator )
{
	INT64 ulIntegerPart, ulDecimalPart;
	INT64 ulNumber;

	ulNumber = _atoi64( (char *) szString );

	ulIntegerPart = ulNumber / 100;
	ulDecimalPart = ulNumber % 100;

	sprintf( (char *) szString, "%013I64d%c%02I64d", ulIntegerPart, uchDecimalSeparator, ulDecimalPart );

	return true;
}

bool bStringToDateHour( unsigned char * pszOut, unsigned char * pszDate, unsigned char * pszTime )
{
	if( pszOut != 0x00 && pszDate != 0x00 && pszTime != 0x00 )
	{
		sprintf( (char *) pszOut,"%.2s/%.2s/%.4s %.2s:%.2s:%.2s", pszDate, pszDate + 2, pszDate + 4, pszTime, pszTime + 2, pszTime + 4 );
	}
	else
		return false;

	return true;
}
int iConvertDecimalPlaces( unsigned char *szSourceValue, unsigned char *szTargetValue,
						   int iTamBuffTarget, int iSourcePlaces, int iTargetPlaces )
{
	if( iSourcePlaces == iTargetPlaces && iTamBuffTarget > (int) strlen((const char *)szSourceValue) )
	{
		strcpy( (char*) szTargetValue, (char*) szSourceValue);
		return SUCCESS;
	}

	// Return Error if the converted value is bigger than the szTargetValue buffer
	if( iTamBuffTarget < (int) ( strlen( (char *) szSourceValue ) - iSourcePlaces + iTargetPlaces + 1 ) )
		return ERROR;

	if( iSourcePlaces < iTargetPlaces  )
	{
		sprintf( (char *) szTargetValue, "%s%0*d", szSourceValue, abs( iSourcePlaces - iTargetPlaces ), 0 );
		return SUCCESS;
	}
	else
	{
		memcpy(szTargetValue, szSourceValue, strlen( (char *) szSourceValue ) - abs( iSourcePlaces - iTargetPlaces ) );
		szTargetValue[ strlen( (char *) szSourceValue ) - abs( iSourcePlaces - iTargetPlaces ) ]=0x00;
		return SUCCESS;
	}
}

/* Current Formats Accepted - ddmmaa, dd/mm/aa, ddmmaaaa and dd/mm/aaaa */
int iCheckDateField( unsigned char * szDate, int * iDateFormat )
{
	unsigned char szDay[3], szMonth[3], szYear[5];
	int iCounter;

	memset( szDay, 0x00, sizeof( szDay ) );
	memset( szMonth, 0x00, sizeof( szMonth ) );
	memset( szYear, 0x00, sizeof( szYear ) );

	switch( strlen( (char *) szDate ) )
	{
		case 0x06:
			for( iCounter = 0; iCounter < 0x06; iCounter++ )
			{
				if( !isdigit( szDate[iCounter] ) )
					return INTERNAL_CHECK_ERROR;
			}

			sprintf( (char *) szDay, "%.2s", szDate );
			sprintf( (char *) szMonth, "%.2s", szDate + 2 );
			sprintf( (char *) szYear, "20%.2s", szDate + 4 );
			
			*iDateFormat = INTERNAL_DATE_FORMAT_DDMMAA;
			break;

		case 0x08:
		{
			int iType = 0;

			if( szDate[2] == '/' )
				iType = 1;

			for( iCounter = 0; iCounter < 0x08; iCounter++ )
			{
				if( (iCounter == 2 || iCounter == 5 ) && iType == 1 )
				{
					if( szDate[iCounter] != '/' )
						return INTERNAL_CHECK_ERROR;
				}
				else
				{
					if( !isdigit( szDate[iCounter] ) )
						return INTERNAL_CHECK_ERROR;
				}
			}

			if( iType == 1 )
			{
				sprintf( (char *) szDay, "%.2s", szDate );
				sprintf( (char *) szMonth, "%.2s", szDate + 3 );
				sprintf( (char *) szYear, "20%.2s", szDate + 6 );	
				*iDateFormat = INTERNAL_DATE_FORMAT_DD_MM_AA;
			}
			else
			{
				sprintf( (char *) szDay, "%.2s", szDate );
				sprintf( (char *) szMonth, "%.2s", szDate + 2 );
				sprintf( (char *) szYear, "%.4s", szDate + 4 );	
				*iDateFormat = INTERNAL_DATE_FORMAT_DDMMAAAA;
			}

			break;
		}

		case 0x0a:
			for( iCounter = 0; iCounter < 0x0a; iCounter++ )
			{
				if( iCounter == 2 || iCounter == 5 )
				{
					if( szDate[iCounter] != '/' )
						return INTERNAL_CHECK_ERROR;
				}
				else
				{
					if( !isdigit( szDate[iCounter] ) )
						return INTERNAL_CHECK_ERROR;
				}
			}

			sprintf( (char *) szDay, "%.2s", szDate );
			sprintf( (char *) szMonth, "%.2s", szDate + 3 );
			sprintf( (char *) szYear, "%.4s", szDate + 6 );

			*iDateFormat = INTERNAL_DATE_FORMAT_DD_MM_AAAA;
			break;

		default:
			return INTERNAL_CHECK_ERROR;
	}

	if( atoi( (char *) szMonth ) < 1 || atoi( (char *) szMonth ) > 12 ||
		atoi( (char *) szDay )   < 1 || atoi( (char *) szDay ) > iMonthDays( szMonth, szYear ) ||
		atoi( (char *) szYear )  < 1 )
	{
		return INTERNAL_CHECK_ERROR;
	}

	return INTERNAL_CHECK_SUCCESS;
}

int iConvertTime( unsigned char *szTime, int iFormat )
{
	unsigned char szTimeAux[16];
	int iSourceFormat;

	if( iCheckTimeField( szTime, &iSourceFormat ) == INTERNAL_CHECK_ERROR )
		return ERROR;

	if( iSourceFormat == iFormat )
		return SUCCESS;

	strcpy( (char *) szTimeAux, (char *) szTime );

	switch( iSourceFormat )
	{
		case INTERNAL_TIME_FORMAT_HHMMSS:	
			sprintf( (char *) szTime, "%.2s:%.2s:%.2s", szTimeAux, szTimeAux + 2, szTimeAux + 4 );
			break;

		case INTERNAL_TIME_FORMAT_HH_MM_SS:	
			sprintf( (char *) szTime, "%.2s%.2s%.2s", szTimeAux, szTimeAux + 3, szTimeAux + 6 );
			break;

		default:
			return ERROR;
	}

	return SUCCESS;
}

int iConvertDate( unsigned char *szDate, int iFormat )
{
	unsigned char szDateAux[16];	
	int iSourceFormat;
	
	if( iCheckDateField( szDate, &iSourceFormat ) == INTERNAL_CHECK_ERROR )
		return ERROR;

	if( iSourceFormat == iFormat )
		return SUCCESS;

	strcpy( (char *) szDateAux, (char *) szDate );

	switch( iSourceFormat )
	{
		case INTERNAL_DATE_FORMAT_DDMMAA:
			switch( iFormat )
			{
				case INTERNAL_DATE_FORMAT_DD_MM_AA:
					sprintf( (char *) szDate, "%.2s/%.2s/%.2s", szDateAux, szDateAux + 2, szDateAux + 4 );
					break;

				case INTERNAL_DATE_FORMAT_DDMMAAAA:
					sprintf( (char *) szDate, "%.4s20%.2s", szDateAux, szDateAux + 4 );
					break;

				case INTERNAL_DATE_FORMAT_DD_MM_AAAA:
					sprintf( (char *) szDate, "%.2s/%.2s/20%.2s", szDateAux, szDateAux + 2, szDateAux + 4 );
					break;

				default: 
					return ERROR;
			}
			break;

		case INTERNAL_DATE_FORMAT_DD_MM_AA:
			switch( iFormat )
			{
				case INTERNAL_DATE_FORMAT_DDMMAA:
					sprintf( (char *) szDate, "%.2s%.2s%.2s", szDateAux, szDateAux + 3, szDateAux + 6 );
					break;

				case INTERNAL_DATE_FORMAT_DDMMAAAA:
					sprintf( (char *) szDate, "%.2s%.2s20%.2s", szDateAux, szDateAux + 3, szDateAux + 6 );
					break;

				case INTERNAL_DATE_FORMAT_DD_MM_AAAA:
					sprintf( (char *) szDate, "%.6s20%.2s", szDateAux, szDateAux + 6 );
					break;

				default:
					return ERROR;
			}
			break;

		case INTERNAL_DATE_FORMAT_DDMMAAAA:
			switch( iFormat )
			{
				case INTERNAL_DATE_FORMAT_DDMMAA:
					sprintf( (char *) szDate, "%.4s%.2s", szDateAux, szDateAux + 6 );
					break;

				case INTERNAL_DATE_FORMAT_DD_MM_AA:
					sprintf( (char *) szDate, "%.2s/%.2s/%.2s", szDateAux, szDateAux + 2, szDateAux + 6 );
					break;

				case INTERNAL_DATE_FORMAT_DD_MM_AAAA:
					sprintf( (char *) szDate, "%.2s/%.2s/%.4s", szDateAux, szDateAux + 2, szDateAux + 4 );
					break;

				default: 
					return ERROR;
			}
			break;

		case INTERNAL_DATE_FORMAT_DD_MM_AAAA:
			switch( iFormat )
			{
				case INTERNAL_DATE_FORMAT_DDMMAA:
					sprintf( (char *) szDate, "%.2s%.2s%.2s", szDateAux, szDateAux + 3, szDateAux + 8 );
					break;

				case INTERNAL_DATE_FORMAT_DD_MM_AA:
					sprintf( (char *) szDate, "%.6s%.2s", szDateAux, szDateAux + 8 );
					break;

				case INTERNAL_DATE_FORMAT_DDMMAAAA:
					sprintf( (char *) szDate, "%.2s%.2s%.4s", szDateAux, szDateAux + 3, szDateAux + 6 );
					break;

				default: 
					return ERROR;
			}
			break;

		default: 
			return ERROR;
	}

	return SUCCESS;
}

/* Current Formats Accepted - hh:mm:ss and hhmmss */
int iCheckTimeField( unsigned char * szTime, int * iTimeFormat )
{
	unsigned char szHours[3], szMinutes[3], szSeconds[3];
	int iCounter;

	memset( szHours, 0x00, sizeof( szHours ) );
	memset( szMinutes, 0x00, sizeof( szMinutes ) );
	memset( szSeconds, 0x00, sizeof( szSeconds ) );
	
	switch( strlen( (char *) szTime) )
	{
		/* Possible Format -> hhmmss */
		case 0x06:
			for( iCounter = 0; iCounter < 0x06; iCounter ++ )
			{
				if( !isdigit( szTime[iCounter] ) )
					return INTERNAL_CHECK_ERROR;
			}

			sprintf( (char *) szHours,"%.2s", szTime );
			sprintf( (char *) szMinutes, "%.2s", szTime + 2 );
			sprintf( (char *) szSeconds, "%.2s", szTime + 4 );

			*iTimeFormat = INTERNAL_TIME_FORMAT_HHMMSS;
			break;

		/* Possible Format -> hh:mm:ss */
		case 0x08:
			for( iCounter = 0; iCounter < 0x08; iCounter ++ )
			{
				if( iCounter == 2 || iCounter == 5 )
				{
					if( szTime[iCounter] != ':' )
						return INTERNAL_CHECK_ERROR;
				}
				else
				{
					if( !isdigit( szTime[iCounter] ) )
						return INTERNAL_CHECK_ERROR;
				}
			}	

			sprintf( (char *) szHours,"%.2s", szTime );
			sprintf( (char *) szMinutes, "%.2s", szTime + 3 );
			sprintf( (char *) szSeconds, "%.2s", szTime + 6 );

			*iTimeFormat = INTERNAL_TIME_FORMAT_HH_MM_SS;
			break;
		default:		
			return INTERNAL_CHECK_ERROR;
	}	

	/* Check Limits */
	if( atoi( (char *) szHours )   < 0 || atoi( (char *) szHours )   > 23 ||
		atoi( (char *) szMinutes ) < 0 || atoi( (char *) szMinutes ) > 59 ||
		atoi( (char *) szSeconds ) < 0 || atoi( (char *) szSeconds ) > 59 )
	{
		return INTERNAL_CHECK_ERROR;
	}

	return INTERNAL_CHECK_SUCCESS;
}

int iAnoBissexto( unsigned int uiAno ) 
{
    /* Retorna se o ano (com 4 dgitos)  bissexto */
    return !(uiAno % 4) && (uiAno % 100 || !(uiAno % 400)) ;
}

int iMonthDays( unsigned char * szMonth, unsigned char * szYear )
{
	unsigned int vtMonthDays[12] = { 31 , 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

	if( iAnoBissexto( atoi( (char *) szYear ) ) && atoi( (char *) szMonth ) == 2 ) 
		return vtMonthDays[ atoi( (char *) szMonth ) - 1 ] + 1;

	return vtMonthDays[ atoi( (char *) szMonth ) - 1 ];
}

int iConvertCPFCNPJ( unsigned char *szString )
{
	int iCounter;
	unsigned char szAux[32];

	memset( szAux, 0x00, sizeof( szAux ) );

	switch( strlen( ( char *) szString ) )
	{
		/* CPF in format XXXYYYZZZDD */
		case 0x0B:
			for( iCounter = 0; iCounter < 0x0B; iCounter++ )
				if( !isdigit(szString[iCounter]) )
					return INTERNAL_CHECK_ERROR;

			return INTERNAL_CHECK_SUCCESS;

		/* CPF in format XXXYYYZZZ-DD */
		case 0x0C:
			for( iCounter = 0; iCounter < 0x0C; iCounter++ )
				if( iCounter == 9 )
				{
					if( szString[iCounter] != '-' )
						return INTERNAL_CHECK_ERROR;
				}
				else
					if( !isdigit( szString[iCounter] ) )
						return INTERNAL_CHECK_ERROR;
 
			sprintf( (char *) szAux,"%.9s%.2s", szString, szString + 10 );
			strcpy( (char *) szString, (char *) szAux );

			return INTERNAL_CHECK_ERROR;

		/* CPF in format XXX.YYY.ZZZ-DD or CNPJ in format XXYYYZZZAAAADD */
		case 0x0E:
		{
			int iType = 0;

			if( szString[3] == '.' )
				iType = 1;

			for( iCounter = 0; iCounter < 0x0E; iCounter ++ )
				if( ( iCounter == 3 || iCounter == 7 || iCounter == 11) && iType == 1 )
				{
					if( ( iCounter == 3 || iCounter == 7 ) && szString[iCounter] != '.' ||
						  iCounter == 11 && szString[iCounter] != '-' )
					{
						return INTERNAL_CHECK_ERROR;
					}
				}
				else
					if( !isdigit( szString[iCounter] ) )
						return INTERNAL_CHECK_ERROR;

			if( iType == 1 )
			{
				sprintf( (char *) szAux,"%.3s%.3s%.3s%.2s", szString, szString + 4, szString + 8, szString + 12 );
				strcpy( (char *) szString, (char *) szAux );
			}

			return INTERNAL_CHECK_SUCCESS;
		}

		/* CNPJ in Format XXYYYZZZAAAA-DD */
		case 0x0F:
			for( iCounter = 0; iCounter < 0x0F; iCounter ++ )
				if( iCounter == 12 )
				{
					if( szString[iCounter] != '-' )
						return INTERNAL_CHECK_ERROR;
				}
				else
					if( !isdigit( szString[iCounter] ) )
						return INTERNAL_CHECK_ERROR;

			sprintf( (char *) szAux, "%.12s%.2s ", szString, szString + 13 );
			strcpy( (char *) szString, (char *) szAux );
			return INTERNAL_CHECK_SUCCESS;

		case 0x12:
			for( iCounter = 0; iCounter < 0x12; iCounter ++ )
				if( iCounter == 2 || iCounter == 6 || iCounter == 10 || iCounter == 15 )
				{
					if( (iCounter == 2 || iCounter == 6) && szString[iCounter] != '.' )
						return INTERNAL_CHECK_ERROR;

					if( iCounter == 10 && szString[iCounter] != '/' )
						return INTERNAL_CHECK_ERROR;

					if( iCounter == 15 && szString[iCounter] != '-' )
						return INTERNAL_CHECK_ERROR;
				}
				else
					if( ! isdigit( szString[iCounter] ) )
						return INTERNAL_CHECK_ERROR;

			sprintf( (char *) szAux, "%.2s%.3s%.3s%.4s%.2s", szString, szString + 3, szString + 7, szString + 11, szString + 16 );
			strcpy( (char *) szString, (char *) szAux );
			return INTERNAL_CHECK_SUCCESS;

		default:
			return INTERNAL_CHECK_ERROR;
	}

	return INTERNAL_CHECK_SUCCESS;
}

/* Receives CPF in Format XXXYYYZZZDD */
int iCheckCPFCNPJ( unsigned char * szInput )
{
	int iCounter, iBasis = 2, iTotal = 0, iTurn = 0;
	unsigned char szDigits[3], szInputAux[32];
	int iType = 0, iStartPosition;

	memset( szDigits, 0x00, sizeof( szDigits ) );
	memset( szInputAux, 0x00, sizeof( szInputAux ) );

	if( strlen( (char *) szInput ) == 11 )
		iType = 1;

	if( iType == 1 )
		sprintf( (char *) szInputAux, "%.9s", szInput );
	else
		sprintf( (char *) szInputAux, "%.12s", szInput );

	for( iTurn = 0; iTurn < 2; iTurn++ )
	{
		iTotal = 0;
		iBasis = 2;

		iStartPosition = (int) strlen( (char *) szInputAux );
		for( iCounter = iStartPosition + iTurn - (1+iTurn); iCounter >= 0; iCounter-- )
		{
			iTotal += (szInputAux[iCounter] - 0x30) * iBasis++;

			if( iType == 0 && iBasis > 9 )
				iBasis = 2;
		}

		iTotal %= 11;

		if( iTotal == 0 || iTotal == 1 )
			szDigits[iTurn] = 0x30;
		else
			szDigits[iTurn] =  11 - iTotal + 0x30;

		szInputAux[ iStartPosition ] = szDigits[iTurn];
	}

	if( strcmp( (char *) szInput, (char *) szInputAux ) )
		return ERROR;
	else
		return SUCCESS;
}

void vCodeGrandTotal( unsigned char * pszGT, unsigned char * pszCode )
{
	unsigned char szGTAux[19];
	int iCounter;

	sprintf( (char *) szGTAux,"%018ld", (long) atoi( (char *) pszGT ) );

	for( iCounter = 0; iCounter < 18 ; iCounter++ )
	{
		szGTAux[ iCounter ] = pszCode[ szGTAux[iCounter] - 0x30 ];
	}

	strcpy( (char *) pszGT, (char *) szGTAux );
}

void vReplaceChar( unsigned char * szString, unsigned char uchSource, unsigned char uchTarget )
{
	int iCounter;

	for( iCounter = 0; iCounter < (int) strlen( (char *) szString ); iCounter ++ )
	{
		if( szString[iCounter] == uchSource )
			szString[iCounter] = uchTarget;
	}
}

unsigned char * pszSkipLeadingZeros( unsigned char * szString )
{
	int iCounter = 0;
	int iLength = (int) strlen( (char *) szString ) - 1;

	while( szString[0] == 0x30 && iCounter < iLength && szString[1] != '.' && szString[1] != ',' )
	{
		szString++;
		iCounter++;
	}
	
	return szString;
}

// MateusPL apr 20 2005
void vConvertBemaToEpsonMedia( unsigned char *pszMedia, int iDecPlaces )
{
	double  fVal;
	unsigned int		iCounter;
	
	for(iCounter=0; iCounter<strlen((char*)pszMedia); iCounter++)
		if( pszMedia[iCounter] == ',' )
			pszMedia[iCounter] = '.';

	fVal =  atof( (char*)pszMedia );
	fVal *= iDecPlaces==2?100:1000;

	sprintf( (char*)pszMedia, "%d", (int) fVal );
}
// MateusPL end


// MateusPL begin may 18 2005
// Convert text file ro Rich Text
// Returns	0 - ok
//			1 - input file error - not found, cannot open, etc
//			2 - output file error - cannot create
// Attention: This function overwrites the output file.
int iConvertTxtToRtf( char *szTxtFile, char *szRtfFile )
{
	FILE	*fInput, *fOutput;
	char	szLine[1510];
	int		iCounter1, iCounter2;

	// Open and check results to input file
	if( ( fInput = fopen( szTxtFile, "rt" ) ) == NULL )
		return 1;

	// Open and check results to output file
	if( ( fOutput = fopen( szRtfFile, "wt" ) ) == NULL )
	{
		fclose( fInput );
		return 2;
	}

	// Write the Rtf header to the output file
	sprintf( szLine, "%s\n", 
		"{\\rtf1\\ansi\\ansicpg1252\\deff0\\deflang1046{\\fonttbl{\\f0\\fmodern\\fprq1\\fcharset0 Courier New;}{\\f1\\fswiss\\fcharset0 Arial;}}" );
	fwrite( szLine, strlen(szLine), 1, fOutput );
	
	sprintf( szLine, "%s", 
		"\\viewkind4\\uc1\\pard\\f0\\fs20 " );
	fwrite( szLine, strlen(szLine), 1, fOutput );
    
	// For each line of input file, write the output with Rtf additional information
	while( fgets( szLine, 1500, fInput ) )
	{
		// Some ajusts in the text line
		szLine[ strlen( szLine ) - 1 ] = 0x00;
		// Find the symbol '{' and '}' and put an back slash before it
		for( iCounter1=0; szLine[ iCounter1 ]; iCounter1++ )
		{
			if( szLine[ iCounter1 ] == '{' || szLine[ iCounter1 ] == '}' )
			{
				szLine[ strlen(szLine) + 1 ] = 0x00;
				for( iCounter2=(int)strlen(szLine); iCounter2>iCounter1; iCounter2-- )
				{
					szLine[ iCounter2 ] = szLine[ iCounter2 - 1 ];
				}
				szLine[ iCounter1 ] = '\\';
				iCounter1++;
			}
		}

		strcat( szLine, "\\par\n" );
		fwrite( szLine, strlen(szLine), 1, fOutput );
	}

	// Write the Rtf footer to the output file
	sprintf( szLine, "%s\n", 
		"}" );
	fwrite( szLine, strlen(szLine), 1, fOutput );

	// Close files
	fclose( fInput );
	fclose( fOutput );
	// Finished with success
	return 0;
}
// MateusPL end


// MateusPL jul 12 2005
// Build the Barcode sequence
int iBuildBarCodeSequence( INT iType, INT iHeight, INT iWidth, INT iPosition, INT iCharType, LPSTR szCode, LPSTR szCodFormated )
{
	int iPos=0;

	// Start of the BarCode
	szCodFormated[iPos++] = 0x1b;
	// Special Text Attributes
	szCodFormated[iPos++] = (char) 0x80; // Barcode
	// BarCode Type
	szCodFormated[iPos++] = (char) iType;
	// Height
	szCodFormated[iPos++] = (char) iHeight;
	// Width
	szCodFormated[iPos++] = (char) iWidth;
	// Text Position
	szCodFormated[iPos++] = (char) iPosition;
	// Font Style
	szCodFormated[iPos++] = (char) ( iCharType + 48 );
	sprintf( (char*) szCodFormated+iPos, "%s", (char*) szCode);
	iPos+=(int)strlen( (char*) szCode );

	return iPos;
}
// MateusPL end

bool if_true( char chFlag )
{
	if( chFlag=='T' || chFlag=='S' || chFlag=='V' || 
		chFlag=='t' || chFlag=='s' || chFlag=='v' || 
		chFlag=='1' )
		return true;

	return false;
}


INT iCreatePath(CHAR * szOriginalPath)
{
	CHAR szPath[1001];
	CHAR szPathAux[1001];
	CHAR *szEnd;
	INT  iSize;

	memset(szPath,0,sizeof(szPath));
	strncpy(szPath,szOriginalPath,1000);

	if((szPath == NULL) || (!strcmp(szPath,"")))
		return ERROR;

	if(_mkdir(szPath) == 0)
		return SUCCESS;
	else
	{
		if(errno == EEXIST)
			return SUCCESS;
		else if (errno != ENOENT)
			return ERROR;
	}
	
	memset(szPathAux,0,sizeof(szPathAux));	
	szEnd = strrchr(szPath,'\\');

        
	iSize = (int) (szEnd-szPath);
	strncpy(szPathAux,szPath,iSize);
	szPathAux[szEnd-szPathAux]=0;

	if (iCreatePath(szPathAux) == SUCCESS)
		return iCreatePath(szPath);
	else
		return ERROR;

}	

INT iGetEpsonPath(CHAR * szAppPath)
{
	HKEY	m_Key;
	BYTE	*X;
	DWORD	dwSize, dwType;// dwPath;
	CHAR	szPath[1001];

	if( RegOpenKeyEx( HKEY_CURRENT_USER, "SOFTWARE\\EPSON\\InterfaceEpson", 0x00, KEY_ALL_ACCESS, &m_Key ) == ERROR_SUCCESS )
	{
		dwSize=1001; dwType=REG_SZ;
		X = (BYTE *) &szPath;
			if(	RegQueryValueEx( m_Key, "Path", 0x00, &dwType , X, &dwSize ) == ERROR_SUCCESS )
			{
				RegCloseKey( m_Key );
				strcat(szPath, "\\");

				if((isalpha(szPath[0]) != 0) && (szPath[1] == ':'))
				{
					if(iCreatePath(szPath) ==  SUCCESS)
					{
						strcpy(szAppPath,szPath);
						strcat(szAppPath, "\\");
						return 0;
					}
				}
			}

    }
 
	if(iCreatePath("C:\\Epson\\InterfaceEpson\\DataFiles\\") ==  SUCCESS)
	{
		strcpy(szAppPath, "C:\\Epson\\InterfaceEpson\\DataFiles\\");
	}
	else
	{
		strcpy(szAppPath, "C:\\");
	}
	
	//Write folder on register
	if( RegOpenKeyEx( HKEY_CURRENT_USER, "SOFTWARE\\EPSON\\InterfaceEpson", 0x00, KEY_ALL_ACCESS, &m_Key ) != ERROR_SUCCESS )
	{
		RegCreateKeyEx( HKEY_CURRENT_USER, "SOFTWARE\\EPSON\\InterfaceEpson", 0x00, NULL, REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &m_Key, NULL );		
	}
	X = (BYTE *) szAppPath;
	RegSetValueEx( m_Key, "Path", 0x00, REG_SZ, X , ((DWORD)strlen(szAppPath)+1) );
	RegCloseKey( m_Key );

	return 0;

}

bool if_false( char chFlag )
{
	return ( ! if_true( chFlag ) );
}
