/***********************************************************************/
/* SERIAL.CPP :                                                        */
/*      Contains Prototypes and Defines used to send and receive data  */
/*  throught the serial port                                           */
/***********************************************************************/

#include "stdafx.h"
#include "utils.h"
#include "serial.h"
#include "state.h"
#include <iostream>

static CSerial clsSerial_Host;		/* Instance to handle the Host port		*/

bool bExecThread = false; // Controls the execussion of the thread code

BYTE szACK[2] = { ACK_CHAR, 0x00 }; /* Defines the ACK String to be sent by the end of the command */
BYTE szNCK[2] = { NCK_CHAR, 0x00 }; /* Defines the NACK String to be sent by the end of the command */

#ifdef COMPILE_LOG_SERIAL
// Set Log Parameters
void _vSerial_Set_Log( char * szUserParh, char chUserAction )
{	
	clsSerial_Host.chAction = chUserAction;
	clsSerial_Host.clsLog.SetPath( szUserParh );
}
#endif

int _iSerial_Send_Buffer( unsigned char * szBuffer, int iBufferSize )
{	
	return clsSerial_Host.iCSerial_Send_Buffer( szBuffer, iBufferSize );
}

int _iSerial_Get_Comm_Status( void )
{
	return clsSerial_Host.iCSerial_Get_Comm_Status();
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Open_Port                                    */
/*                                                                      */
/* Parameters   : pstSerial_Conf                                        */
/*					  Pointer to a Serial Configuration Structure       */
/*                                                                      */
/* Returns      : SERIAL_PORT_OPEN_OK or ERROR_SERIAL_XXX               */
/************************************************************************/
/* Description  : This function is the entry point to the serial        */
/*                open port process. It calls another function that     */
/*                setups the communication port and opens it.           */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 23/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Open_Port( PSTSERIAL_CONFIG pstSerial_Conf )
{
	return clsSerial_Host.iCSerial_Open_Serial_Port( pstSerial_Conf );
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Close_Port                                   */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_PORT_NOT_OPEN         */
/************************************************************************/
/* Description  : This function closes the serial port                  */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Close_Port( void )
{
	return clsSerial_Host.iCSerial_Close_Serial_Port();
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Send_Message                                 */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_XXX                   */
/************************************************************************/
/* Description  : This function sends the message through the serial    */
/*                port													*/
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Send_Message( void )
{
	if( _iSerial_Get_Comm_State() == SERIAL_STATE_CLOSED )
		return ERROR_SERIAL_PORT_NOT_OPEN;

	clsSerial_Host.vCSerial_Set_Comm_State( SERIAL_STATE_BUSY );

	// Insted of thread, use plain code.
	
	int iReturn; // Function Return Code

	// Send the message calling the respective class function
	iReturn = clsSerial_Host.iCSerial_Send_Message();

	clsSerial_Host.vCSerial_Set_Last_Comm_Error( iReturn );
	clsSerial_Host.vCSerial_Set_Comm_State( SERIAL_STATE_OPEN );

	// Purge the Input Buffer to avoid overwriting.
	clsSerial_Host.vCSerial_Purge_Message();
	
	return SERIAL_NO_ERROR;
}

int _iSerial_Resend_Message( void )
{
	clsSerial_Host.vCSerial_Recover_Message();

	return _iSerial_Send_Message();
}

int _iSerial_Get_Last_Comm_Error( void )
{
	return clsSerial_Host.iCSerial_Get_Last_Comm_Error();
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Add_FieldEx                                  */
/*                                                                      */
/* Parameters   : szField												*/
/*						Pointer to a buffer with the Field to be joint	*/
/*                iFieldLength											*/
/*						The buffer's size								*/
/*                uiFieldType											*/
/*						Field type to be added							*/
/*                bSeparator                                            */
/*						Add or not the separator at the Field's end		*/
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_XXX                   */
/************************************************************************/
/* Description  : This function adds the buffer to the internal message */
/*                buffer.												*/
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Add_FieldEx( unsigned char *  szField, size_t iFieldLength, unsigned int uiFieldType, bool bSeparator )
{
	if( szField == NULL )
		return ERROR_SERIAL_PARAMETER_NULL;

	return clsSerial_Host.iCSerial_Add_FieldEx( szField, iFieldLength, uiFieldType, bSeparator );
}

int _iSerial_Get_Comm_State( void )
{
	return clsSerial_Host.iCSerial_Get_Comm_State();
}

void CSerial::vCSerial_Set_Last_Comm_Error( int iError )
{
	iLastCommError = iError;
}

int CSerial::iCSerial_Get_Last_Comm_Error( void )
{


	return iLastCommError;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Add_Field                                    */
/*                                                                      */
/* Parameters   : szField												*/
/*						Pointer to a buffer with the Field to be joint	*/
/*                iFieldLength											*/
/*						The buffer's size								*/
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_XXX                   */
/************************************************************************/
/* Description  : This function adds the buffer to the internal message */
/*                buffer.												*/
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Add_Field( unsigned char *  szField, size_t iFieldLength )
{
	if( szField == NULL )
		return ERROR_SERIAL_PARAMETER_NULL;

	return clsSerial_Host.iCSerial_Add_Field( szField, iFieldLength );
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _vSerial_Purge_Message                                */
/*                                                                      */
/* Parameters   : None   												*/
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_XXX                   */
/************************************************************************/
/* Description  : This function empties the internal message buffer     */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
void _vSerial_Purge_Message( void )
{
	clsSerial_Host.vCSerial_Purge_Message();
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Get_FieldEx                                  */
/*                                                                      */
/* Parameters   : vAnswer  												*/
/*                    Pointer to hold the function's final answer		*/
/*                iAnswerField                                          */
/*                    Host answer field's number						*/
/*                iAnswerType											*/
/*					  Answer's type desired								*/
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_XXX                   */
/************************************************************************/
/* Description  : This function gets a field inside the final answer    */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Get_FieldEx( void * vAnswer, int *iAnswerSize, int iAnswerField, int iAnswerType )
{
	if( vAnswer == NULL || iAnswerSize == NULL )
		ERROR_SERIAL_PARAMETER_NULL;

	return clsSerial_Host.iCSerial_Get_FieldEx( vAnswer, iAnswerSize, iAnswerField, iAnswerType );
}
/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Get_Field                                    */
/*                                                                      */
/* Parameters   : vAnswer  												*/
/*                    Pointer to hold the function's final answer		*/
/*                iAnswerField                                          */
/*                    Host answer field's number						*/
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_XXX                   */
/************************************************************************/
/* Description  : This function gets a field inside the final answer    */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Get_Field( void *vAnswer, int *iAnswerSize, int iAnswerField )
{
	if( vAnswer == NULL || iAnswerSize == NULL )
		return ERROR_SERIAL_PARAMETER_NULL;

	return clsSerial_Host.iCSerial_Get_Field( vAnswer, iAnswerSize, iAnswerField );
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : _iSerial_Get_Field_Counter                            */
/*                                                                      */
/* Parameters   : None  												*/
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_XXX                   */
/************************************************************************/
/* Description  : This function gets a field inside the final answer    */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int _iSerial_Get_Field_Counter( void )
{
	return clsSerial_Host.iCSerial_Get_Field_Counter();
}

int _iSerial_Get_Sent_Frame( unsigned char * szBuffer, int *iBufferLength )
{
	if( szBuffer == NULL || iBufferLength == NULL )
		return ERROR_SERIAL_PARAMETER_NULL;

	return clsSerial_Host.iCSerial_Get_Sent_Frame( szBuffer, iBufferLength );
}

int _iSerial_Get_Received_Frame( unsigned char * szBuffer, int *iBufferLength )
{
	if( szBuffer == NULL || iBufferLength == NULL )
		return ERROR_SERIAL_PARAMETER_NULL;

	return clsSerial_Host.iCSerial_Get_Received_Frame( szBuffer, iBufferLength );
}



/************************************************************************/
/************************* Class Implementation *************************/
/**************************** DO NOT MODIFY *****************************/
/************************************************************************/

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::CSerial                                      */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : None                                                  */
/************************************************************************/
/* Description  : Class Constructor                                     */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
CSerial::CSerial()
{
	/* Setup the internal buffers' size to zero */
	iSendLength = 0;	
	iRecvLength = 0;
	iAuxLength = 0;
	iAuxReturnLength = 0;
	iMaxDataAllowed = 0;
	iMaxDataAllowedAux = 0;
	iBufferPosition = 0;

	/* Set the Port Ready Status to false */
	bPortReady = false;

	/* Set the handle to invalid */
	hlCom = INVALID_HANDLE_VALUE;

	/* Define the Comm's Initial State to Closed */
	iCommState = SERIAL_STATE_CLOSED;

	/* Set the initial Message Type to type 1 ( Always binary data ) */
	iMessageType = SERIAL_MESSAGE_TYPE_1;
	iMsgAuxType = SERIAL_MESSAGE_TYPE_1;

	/* Fill the buffers with the NULL char */
	memset( szSendMessage, 0x00, sizeof(szSendMessage) );
	memset( szRecvMessage, 0x00, sizeof(szRecvMessage) );
	memset( szAuxMessage, 0x00, sizeof(szAuxMessage) );
	memset( szAuxReturn, 0x00, sizeof(szAuxReturn) );

	#ifdef COMPILE_LOG_SERIAL
		chAction = LOGGER_LOG_NOW;
	#endif
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::~CSerial                                      */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : None                                                  */
/************************************************************************/
/* Description  : Class Destructor                                      */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
CSerial::~CSerial()
{
	if( hlCom != INVALID_HANDLE_VALUE )
	{
		PurgeComm( hlCom, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);

		try {
			CloseHandle( hlCom );
		}
      catch( std::exception & e )
		//catch( CException * e )
		{
         std::cerr << "Caught: " << e.what( ) << std::endl;
         std::cerr << "Type: " << typeid( e ).name( ) << std::endl;
			//e->ReportError();
		}		
	}
}

/* Open Serial Port Should Go Here */
int CSerial::iCSerial_Open_Serial_Port( PSTSERIAL_CONFIG pstSerial_Config )
{
	int iError; /* Variable to handle error codes */

	/* Before opening the port, let's check its current state */
	if( iCSerial_Get_Comm_State() == SERIAL_STATE_OPEN )
		/* If the port is already open, return this state */
		return ERROR_SERIAL_ALREADY_OPEN;

	/* Make all the port's configuration */
	iError = iCSerial_Setup_Comm_Port( pstSerial_Config );

	if( iError != SERIAL_NO_ERROR )
		/* If any error occur during the setup phase, let's close the handle */
		vCSerial_Close_Handle();
	else
	{
		/* Get a random sequence number to start working */
		vSetSequenceNumber();
	}

	/* Return the error occurred during the open port phase */
	return iError;
}

int CSerial::iCSerial_Close_Serial_Port( void )
{
	/* Before closing the port, let's check if it is opened */
	if( iCSerial_Get_Comm_State() == SERIAL_STATE_CLOSED )
		/* If the port is not open, return this state */
		return ERROR_SERIAL_PORT_NOT_OPEN;

	/* Close the handle to avoid resource leak */
	vCSerial_Close_Handle();

	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}
/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Comm_Port                     */
/*                                                                      */
/* Parameters   : pstSerial_Conf                                        */
/*					  Pointer to a Serial Configuration Structure       */
/*                                                                      */
/* Returns      : SERIAL_PORT_OPEN_OK or ERROR_SERIAL_XXX               */
/************************************************************************/
/* Description  : This function setups de communication port and opens  */
/*                it, by calling other member functions.                */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 23/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Setup_Comm_Port( PSTSERIAL_CONFIG pstSerial_Conf )
{
	int iError;			/* Variable to hold the functions' return codes */
	string szComPort;	/* String to be formated with the Com's port number */
	//CString szComPort;	/* String to be formated with the Com's port number */

	/* Convert the port number to the format "COMX" where X is the port number */
	iError = iCSerial_Setup_Comm_Port( pstSerial_Conf->sComm_Port, &szComPort );

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return( iError );

	/* Create the handle to the Serial Port */
	iError = iCSerial_Create_Handle( szComPort );

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return iError;

	/* Setup the buffer size */
	iError = iCSerial_Setup_Buffer();

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return iError;

	/* Setup Communication's parity */
	iError = iCSerial_Setup_Parity( pstSerial_Conf->sParity );

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return iError;

	/* Setup Communication's word length */
	iError = iCSerial_Setup_Data_Bits( pstSerial_Conf->sData_Bits );

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return iError;

	/* Setup Communication's stop bits */
	iError = iCSerial_Setup_Stop_Bits( pstSerial_Conf->sStop_Bits );

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return iError;

	/* Setup Communication's Baud Rate */
	iError = iCSerial_Setup_Baud_Rate( pstSerial_Conf->iBaud_Rate );

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return iError;

	/* Sets abort Communication on error conditions */
	dcb.fAbortOnError = TRUE;

	/* Setup Communication's State */
	iError = iCSerial_Set_Comm_State();

	/* If an error ocurred, return it */
	if( iError != SERIAL_NO_ERROR )
		return iError;

	/* Setup Communication's Timeout */
	iError = iCSerial_Set_Comm_TimeOut();	

	/* If an error ocurred, return it */
	if( iError == SERIAL_NO_ERROR )
		iCommState = SERIAL_STATE_OPEN;

	/* Return the file error code */
	return iError;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Comm_Port                     */
/*                                                                      */
/* Parameters   : sComm_Port                                            */
/*					  Contains the Com Port. Valid Range 1 to 4         */
/*                szComPort                                             */
/*                    A Port to a CString contaning the relative port   */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_COM           */
/************************************************************************/
/* Description  : This function converts the argument passed as an inte */
/*                ger to a CString containing the value "Com" plus the  */
/*                Communication port Number.                            */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 23/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Setup_Comm_Port( short sComm_Port, string * szComPort )
{
	/* Select the desired Com Port and Format it */
	switch( sComm_Port )
	{
		case 1:
			*szComPort = "Com1";
			break;
		case 2:
			*szComPort = "Com2";
			break;
		case 3:
			*szComPort = "Com3";
			break;
		case 4:
			*szComPort = "Com4";
			break;
		default:
			/* Invalid Com - Report it */
			*szComPort = "Invalid Com Port";
			return ERROR_SERIAL_INVALID_COM;
	}

	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Create_Handle                       */
/*                                                                      */
/* Parameters   : szComPort                                             */
/*					  CString containing the Com Port Name              */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_HANDLE        */
/************************************************************************/
/* Description  : This function creates the handle to the Com Port.     */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 23/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/* 06/06/2003    1.1      Serial Changes    Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/* 1.1       Change the File attribute to OVERLAPPED I/O                */
/************************************************************************/
int CSerial::iCSerial_Create_Handle( string szComPort )
{
   hlCom = CreateFile( szComPort.c_str(),										/* File Name                                 */
		 			    GENERIC_READ | GENERIC_WRITE,					/* Access Mode - Open port to Read and Write */
						NULL,											/* Share Mode - 0 = Exclusive Access         */
						NULL,											/* Security Attributes - No Security         */
						OPEN_EXISTING,									/* How to Create the File                    */
						FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,	/* File Attributers - Overlapped I/O        */
						NULL);											/* File Template - Null  = Template          */
	
	/* If the Create File returns an invalid handle, check why this happens to return the correct code */
	if( hlCom == INVALID_HANDLE_VALUE )
	{
		if( GetLastError() == ERROR_FILE_NOT_FOUND )
			return ERROR_SERIAL_INVALID_COM; /* Com Port does not exist */
		else
			return ERROR_SERIAL_IN_USE;	/* Com port in used by another application */
	}

	/* Return Create Handle OK */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Buffer                        */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : SERIAL_PORT_OPEN_OK or ERROR_SERIAL_XXX               */
/************************************************************************/
/* Description  : This function sets the max serial's buffer size.      */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 23/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Setup_Buffer(void)
{
	/* Setup the buffer size */
	bPortReady = SetupComm( hlCom, MAX_BUFFER_LENGTH, MAX_BUFFER_LENGTH );

	/* If an Error occurred, return this situation */
	if(bPortReady == false)
		return ERROR_SERIAL_INVALID_BUFFER_SIZE;

	/* Returns operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Parity                        */
/*                                                                      */
/* Parameters   : sParity                                               */
/*                   Parity to be used with the DCB                     */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_PARITY        */
/************************************************************************/
/* Description  : This function sets the parity in the dcb struct       */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 23/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Setup_Parity( short sParity )
{	
	/* Get the current Comm Port State */
	bPortReady = GetCommState(hlCom, &dcb);

	/* If the user chooses 'p' or 'e', select Even Parity */
	if( sParity == 'p' || sParity == 'P' ||
		sParity == 'e' || sParity == 'E' )
	{
	  dcb.Parity = EVENPARITY;
	}
	else
	  /* If the user chooses 'i' or 'o', select Odd Parity */
	  if ( sParity == 'i' || sParity == 'I' ||
		   sParity == 'o' || sParity == 'O' )
	  {
	  	dcb.Parity = ODDPARITY;
	  }
	  else
	    /* If the user chooses 'n', select No parity */
	    if( sParity == 'n' || sParity == 'N' )
	    {
		  dcb.Parity = NOPARITY;
	    }
		/* If the cases above do not match, return an error */
		else
		{
		  return ERROR_SERIAL_INVALID_PARITY;
		}

	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Data_Bits                     */
/*                                                                      */
/* Parameters   : sData_Bits                                            */
/*                   Data Bits used in the serial communication         */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_DATA_BITS     */
/************************************************************************/
/* Description  : This function sets data bits in the dcb struct        */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Setup_Data_Bits( short sData_Bits )
{
	/* Select the desired word size */
	switch( sData_Bits )
	{
		case 7:
			dcb.ByteSize = 7;
			break;
		case 8:
			dcb.ByteSize = 8;
			break;
		default:
			/* If the word size is incorrect, clear it and return an error */
			dcb.BaudRate = 0;
			return ERROR_SERIAL_INVALID_DATA_BITS;
	}

	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Stop_Bits                     */
/*                                                                      */
/* Parameters   : sStop_Bits                                            */
/*                   Stop Bits used in the serial communication         */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_STOP_BITS     */
/************************************************************************/
/* Description  : This function sets the stop bits in the dcb struct    */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Setup_Stop_Bits( short sStop_Bits )
{
	/* Select the desired stop bits length */
	switch( sStop_Bits )
	{
		case 1:
			dcb.StopBits = ONESTOPBIT;
			break;
		case 2:
			dcb.StopBits = TWOSTOPBITS;
			break;
		default:
			/* If the field is incorrect, clear it and return an error */
			dcb.StopBits = 0;
			return ERROR_SERIAL_INVALID_STOP_BITS;
	}
		
	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Baud_Rate                     */
/*                                                                      */
/* Parameters   : sBaud_Rate                                            */
/*                   Speed used in the serial port                      */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_BAUD_RATE     */
/************************************************************************/
/* Description  : This function sets the baud rate in the dcb struct    */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Setup_Baud_Rate( int iBaud_Rate )
{
	/* Select the Total Timeout depending on the Baud Rate */
	switch( iBaud_Rate )
	{
		case 1200:
			iTotalTimeout = SERIAL_1200_BPS_TIMEOUT;
			break;
		case 2400:
			iTotalTimeout = SERIAL_2400_BPS_TIMEOUT;
			break;
		case 4800:
			iTotalTimeout = SERIAL_4800_BPS_TIMEOUT;
			break;
		case 9600:
			iTotalTimeout = SERIAL_9600_BPS_TIMEOUT;
			break;
		case 19200:
			iTotalTimeout = SERIAL_19200_BPS_TIMEOUT;
			break;
		case 38400:	
			iTotalTimeout = SERIAL_38400_BPS_TIMEOUT;
			break;
		default:
			/* If this field is invalid, clear it and return an error */
			dcb.BaudRate = 0;
			return ERROR_SERIAL_INVALID_BAUD_RATE;
	}

	/* Set the Baud Rate */
	dcb.BaudRate = iBaud_Rate;	

	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Setup_Comm_State                    */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_COMM_STATE    */
/************************************************************************/
/* Description  : This function sets the comm to be used                */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/* 06/06/2003    1.1      DCB Changes       Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/* 1.1		 Added additional configurations to the DCB Structure       */
/************************************************************************/
int CSerial::iCSerial_Set_Comm_State( void )
{
	/* Revision 1.1 - Begin - 06/06/2003 */
	dcb.DCBlength = sizeof(DCB);				/* Fill with the structure size */
	dcb.fBinary = true;							/* Indicate Binary mode transfer, windows do not support non binary transfers*/
	dcb.fRtsControl = RTS_CONTROL_DISABLE;		/* Disable RTS control			*/
	dcb.fOutxDsrFlow = true;					/* Indicates whether the DSR (data-set-ready) signal is monitored for output flow control. */
	dcb.fDtrControl = DTR_CONTROL_HANDSHAKE;	/* Enable DTR Control Handshake */
	/* Revision 1.1 - End - 06/06/2003 */

	/* Set the configuration */
	bPortReady = SetCommState(hlCom, &dcb);

	/* If an error occurred during the Setup phase, return it */
	if( bPortReady == false )
		return ERROR_SERIAL_INVALID_COMM_STATE;

	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Get_Comm_State                      */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : Current Serial Port State                             */
/************************************************************************/
/* Description  : This function gets the current state of the comm port */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Get_Comm_State(void)
{
	/* Return the com state. Nowadays, this state can be the following values */
	/* 0 - SERIAL_STATE_CLOSED -> The communication port is closed			  */
	/* 1 - SERIAL_STATE_OPEN   -> The communication port is opened			  */
	return iCommState;
}

void CSerial::vCSerial_Set_Comm_State( int iNewState )
{
	iCommState = iNewState;
}

int CSerial::iCSerial_Get_Sent_Frame( unsigned char * szBufferSend, int * iLength )
{
	memcpy( szBufferSend, szAuxReturn, iAuxReturnLength );
	*iLength = iAuxReturnLength;

	return SERIAL_NO_ERROR;
}
		
int	CSerial::iCSerial_Get_Received_Frame( unsigned char * szBufferReceived, int * iLength )
{
	if( iRecvLength != 0 )
	{
		szBufferReceived[0] = STX_CHAR;

		memcpy( &szBufferReceived[1], szRecvMessage, iRecvLength );
		*iLength = iRecvLength + 1;
	}
	else
	{
		szBufferReceived[0] = 0x00;
		*iLength = 0;
	}

	return SERIAL_NO_ERROR;
}


/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Set_Comm_TimeOut                    */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR or ERROR_SERIAL_INVALID_TIMEOUT       */
/************************************************************************/
/* Description  : This function sets the communication port timeout     */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/* 06/06/2003    1.1      Timeout Changes   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/* 1.1           This revision changes the Communication Port Timeout   */
/*           configuration. The Write Timeout time was changed to the   */
/*           write corresponding function as well the read total time.  */
/************************************************************************/
int CSerial::iCSerial_Set_Comm_TimeOut( void )
{
	/* Get the current Com Port Timeout */
	bPortReady = GetCommTimeouts (hlCom, &CommTimeouts);

	/* If it is not possible to get the current time out, report this error */
	if( bPortReady == false )
		return ERROR_SERIAL_INVALID_TIMEOUT;

	/* Set the minimum space between two char timeout */
	CommTimeouts.ReadIntervalTimeout = SERIAL_MIN_CHAR_TIMEOUT * 3;

	/* Set the new Communication's Timeout */
	bPortReady = SetCommTimeouts (hlCom, &CommTimeouts);
	
	/* If an error ocurred during this operation, report it */
	if( bPortReady == false )
		return ERROR_SERIAL_INVALID_TIMEOUT;

	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::vCSerial_Close_Handle                        */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : None                                                  */
/************************************************************************/
/* Description  : This function closes the communication port           */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 26/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
void CSerial::vCSerial_Close_Handle( void )
{
	/* Free the handle to avoid resource leak */
	if( hlCom != INVALID_HANDLE_VALUE )
	{
		PurgeComm( hlCom, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);

		CloseHandle(hlCom);
	}

	/* Invalidate Handle */
	hlCom = INVALID_HANDLE_VALUE;

	/* Change the Comunication Port State to closed */
	iCommState = SERIAL_STATE_CLOSED;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Prepare_Message                     */
/*                                                                      */
/* Parameters   : None                                                  */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the STX char and the sequence      */
/*                number to the begging of the message and the ETX char */
/*                to the end of the Message                             */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Prepare_Message( void )
{
	char szAux[MAX_BUFFER_LENGTH];	/* Auxiliary Message */
	int  iCounter;					/* Auxiliary counter */

	szAux[0] = STX_CHAR;				/* Add the STX char in the first position */
	szAux[1] = uchGetSequenceNumber();	/* Get the next sequence number			  */

	/* Copy all the original chars to the aux char */
	for( iCounter = 2; iCounter < (iSendLength + 2); iCounter++ )
	{
		szAux[iCounter] = szSendMessage[iCounter-2];
	}

	/* Add the ETX char to the last position */
	szAux[iCounter] = ETX_CHAR;

	/* Update the internal variable to the current value */
	iSendLength = iCounter + 1;

	/* If the internal value is higher than the limit, return overflow error. */
	if( iSendLength >= MAX_BUFFER_LENGTH)
		return ERROR_SERIAL_BUFFER_OVERFLOW;

	/* Update the Send Message Buffer */
	memcpy( szSendMessage, szAux, iSendLength );

	/* Return operation completed successfully. */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Calculate_CheckSum( void )
{
	/* If the current size + the checksum size is higher than the */
	/* maximum length, return overflow error					  */
	if( (iSendLength + 4) > MAX_BUFFER_LENGTH )
		return ERROR_SERIAL_BUFFER_OVERFLOW;

	/* Calculate the checksum */
  {
	  int iCounter  = 0, 
		    iCheckSum = 0;

	  /* This loop adds all chars in the string */
	  for ( iCounter = 0; iCounter < iSendLength; iCounter++ )
	  {
		  iCheckSum += szSendMessage[iCounter];
	  }

	  /* Convert the Checksum calculated in the previous loop and attach them
	     to the end of the original string
       */
	  for( iCounter = 0; iCounter < 4; iCounter++)
	  {
		  int iAux;

		  /* Shift the data n bits right in order to get then separated */
		  iAux = ( iCheckSum >> ( 12 - 4 * iCounter ) ) & 0x000F;

		  /* If it is a digit, only add the ASCII value of char '0'.
		     If it is not, add the ASCII value of char 'a' and subtract 10 from it */
		  if( iAux < 10 )
			  szSendMessage[ iSendLength ] =  iAux + 0x30 ;
		  else
			  szSendMessage[ iSendLength ] =  iAux + 0x41 - 0x0a ;

		  /* Increment the data lenght */
		  iSendLength++;
	  }
  }

	/* Return operation completed successfully */
	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Send_Message                        */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Send_Message( void )
{
	int iResponse;
	int iAnswerTries = 0;

	/* Save Context */
	iAuxLength = iSendLength;
	iMaxDataAllowedAux = iMaxDataAllowed;
	memcpy( szAuxMessage, szSendMessage, iAuxLength );
	iMsgAuxType = iMessageType;

	if( iMessageType == SERIAL_MESSAGE_TYPE_1 )
		iSendLength--;

	if( iCSerial_Get_Comm_State() == SERIAL_STATE_CLOSED )
		return ERROR_SERIAL_PORT_NOT_OPEN;

	iResponse = iCSerial_Prepare_Message();
	if( iResponse != SERIAL_NO_ERROR )
		return iResponse;

	iResponse = iCSerial_Calculate_CheckSum();
	if( iResponse != SERIAL_NO_ERROR )
		return iResponse;

	memcpy( szAuxReturn, szSendMessage, iSendLength );
	iAuxReturnLength = iSendLength;

	for( int iSendTries = 0; iSendTries<3; iSendTries++ )
	{
		iResponse = iCSerial_Send_Buffer();

		if( iResponse != SERIAL_NO_ERROR )
		{
			if( iResponse == ERROR_SERIAL_OVERLAPPED_EVENT )
			{
				int iAnswer;

				iAnswer = iCSerial_Get_Comm_Status();	
				if( iAnswer != SERIAL_NO_ERROR )
				{
					return iAnswer;
				}
			}

			return iResponse;
		}

		iResponse = iCSerial_Parse_Answer();

		if( iResponse == ERROR_SERIAL_TIMEOUT_ERROR && !szRecvMessage[0] ) // input buffer different of null
			continue;
		if( iResponse != ERROR_SERIAL_NACK_ERROR ) // if it's not NACK answer
			break;
	}

	if( iResponse != SERIAL_NO_ERROR )
	{
		do
		{
			if( iResponse == ERROR_SERIAL_TIMEOUT_ERROR ||
				iResponse == ERROR_SERIAL_OVERLAPPED_EVENT )
			{
				int iAnswer;
      
				// Check the line status
				iAnswer = iCSerial_Get_Comm_Status();
				if( iAnswer != SERIAL_NO_ERROR )
					return iAnswer;

				strcpy( (char *) szSendMessage, (char *) szNCK );
				iSendLength = 1;

				iResponse = iCSerial_Send_Buffer();

				if( iResponse != SERIAL_NO_ERROR )
				{
					if( iResponse == ERROR_SERIAL_OVERLAPPED_EVENT )
					{
						int iAnswer;

			            iAnswer = iCSerial_Get_Comm_Status();
						if( iAnswer != SERIAL_NO_ERROR )
							return iAnswer;
					}

					return iResponse;
				}

				iResponse = iCSerial_Parse_Answer();
			} 
			iAnswerTries++;
		} while( iAnswerTries < 3 && iResponse != SERIAL_NO_ERROR );

		strcpy( (char *) szSendMessage, (char *) szACK );
		iSendLength = 1;

		iCSerial_Send_Buffer();
	}
	else
	{
		strcpy( (char *) szSendMessage, (char *) szACK );
		iSendLength = 1;
		iCSerial_Send_Buffer();
	}

	return iResponse;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Send_Buffer( void )
{
	OVERLAPPED	osWrite = { 0 };
	DWORD		dwWritten;
	DWORD		dwRes;
	int			iRes;
	int			iTimeout;

	// Log Serial Information
	#ifdef COMPILE_LOG_SERIAL
		clsLog.LogData( "LLO", szSendMessage, iSendLength, chAction );
	#endif
	// End of log Serial

	/* Create this write operation's OVERLAPPED structure's hEvent */
	osWrite.hEvent = CreateEvent( NULL, TRUE, FALSE, NULL );
	
	if( osWrite.hEvent == NULL )
	{
		/* Error Creating Overlapped event handle */
		return ERROR_SERIAL_OVERLAPPED_EVENT; 
	}

	/* Issue Write */
	if( !WriteFile( hlCom, szSendMessage, iSendLength, &dwWritten, &osWrite ) )
	{	
		if( GetLastError() != ERROR_IO_PENDING )
		{
			/* WriteFile failed, but isn't delayed. Report error and abort. */
			iRes = ERROR_SERIAL_SEND_ERROR;
		}
		else
		{
			iTimeout = iTotalTimeout * iSendLength + WRITE_OFF_TIMEOUT;

			/* Write is pending. */
			dwRes = WaitForSingleObject( osWrite.hEvent, iTimeout );

			switch( dwRes )
			{
				/* Overlapped structure's event has been signaled. */
				case WAIT_OBJECT_0:
					if( !GetOverlappedResult( hlCom, &osWrite, & dwWritten, FALSE ) )
						iRes = ERROR_SERIAL_SEND_ERROR;
					else
						if( dwWritten != iSendLength )
						{
							CloseHandle( osWrite.hEvent );
							return ERROR_SERIAL_TIMEOUT_ERROR;
						}

						iRes = SERIAL_NO_ERROR;
					break;
				default:
					/* An error has occurred in WaitForSingleObject.	*/
					/* This usually indicates a problem with the		*/
					/* OVERLAPPED structure's event handle.				*/
					iRes = ERROR_SERIAL_OVERLAPPED_EVENT;
					break;
			}
		}
	}
	else
	{
		if( dwWritten != iSendLength )
		{
			CloseHandle( osWrite.hEvent );
			return ERROR_SERIAL_TIMEOUT_ERROR;
		}

		/* WriteFile completed immediately. */
		iRes = SERIAL_NO_ERROR;
	}

	CloseHandle( osWrite.hEvent );
	return iRes;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Get_Comm_Status( void )
{
	DWORD	dwModemStatus;
	BOOL	fCTS, fDSR;

	if( !SetCommMask( hlCom, EV_RXCHAR | EV_CTS | EV_DSR | EV_RLSD  ) )
		return ERROR_SERIAL_SET_COM_MASK;

	if( !GetCommModemStatus( hlCom, &dwModemStatus ) )
		return ERROR_SERIAL_GET_MODEM_STATUS;

	fCTS = MS_CTS_ON & dwModemStatus;
	fDSR = MS_DSR_ON & dwModemStatus;

	// Changed (06/06/2006) for Hawk 2
	//if( !fCTS || ! fDSR )
	if( !fCTS && ! fDSR )
		return ERROR_SERIAL_OFF_OR_DISCONNECTED;

	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Receive_Buffer( unsigned char *uchByteRead )
{
	DWORD			dwRead;
	BOOL			fWaitingOnRead = FALSE;
	OVERLAPPED		osReader = { 0 };	
	DWORD dwRes;

	/* Create the overlapped event. Must be closed before exiting	*/
	/* to avoid a handle leak.										*/
	osReader.hEvent = CreateEvent( NULL, TRUE, FALSE, NULL );

	if( osReader.hEvent == NULL )
	{
		/* Error creating overlapped event; abort. */
		return ERROR_SERIAL_OVERLAPPED_EVENT;
	}

	if( !fWaitingOnRead )
	{
		/* Issue read operation */
		if( !ReadFile( hlCom, uchByteRead, 1 , &dwRead, &osReader) )
		{
			if( GetLastError() != ERROR_IO_PENDING ) /* Read not delayed? */
			{
				CloseHandle( osReader.hEvent );
				/* Error in Communications, report it. */
				return ERROR_SERIAL_RECEIVE_ERROR;
			}
			else
				fWaitingOnRead = TRUE;
		}
		else
		{
			CloseHandle( osReader.hEvent );

			if( dwRead != 1 )
				return ERROR_SERIAL_TIMEOUT_ERROR;
			
			
			/* Read completed immediately */
			return SERIAL_NO_ERROR;
		}
	}

	if( fWaitingOnRead )
	{
		dwRes = WaitForSingleObject( osReader.hEvent, READ_TIMEOUT );
		switch( dwRes )
		{
			/* Read completed */
			case WAIT_OBJECT_0:
				if( !GetOverlappedResult( hlCom, &osReader, &dwRead, false ) )
				{
					CloseHandle( osReader.hEvent );

					/* Error in communications, report it. */
					return ERROR_SERIAL_RECEIVE_ERROR;
				}
				else
				{
					CloseHandle( osReader.hEvent );

					/* A byte was not read. Return Timeout */
					if( dwRead != 1 )
						return ERROR_SERIAL_TIMEOUT_ERROR;

					/* Read completed immediately */
					return SERIAL_NO_ERROR;
				}
				/* Reset flag so that another operation can be issued. */
				fWaitingOnRead = FALSE;
				break;

			case WAIT_TIMEOUT:
			case WAIT_FAILED:
				/* Operation is not completed yet. fWaitingOnRead flag is not	*/
				/* changed since I'll loop back around, and I don't want		*/
				/* to issue another read until the first one finishes.			*/
				/*																*/
				/* This is a good time to do some background work.				*/
				CloseHandle( osReader.hEvent );
				return ERROR_SERIAL_IN_USE;

			default:
				/* Error in the WaitForSingleObject; abort.						*/
				/* This indicates a problem with the OVERLAPPED structure's		*/
				/* event handle. */
				CloseHandle( osReader.hEvent );
				return ERROR_SERIAL_INTERNAL_ERROR;
				break;
		}
	}
	
	CloseHandle( osReader.hEvent );
	return SERIAL_NO_ERROR;
}


int CSerial::iCSerial_Receive_Buffer( unsigned char * pszBuffer, int * iBufferSize )
{
	DWORD			dwRead;
	BOOL			fWaitingOnRead = FALSE;
	OVERLAPPED		osReader = { 0 };	
	DWORD			dwAnswer;

	/* Create the overlapped event. Must be closed before exiting	*/
	/* to avoid a handle leak.										*/
	osReader.hEvent = CreateEvent( NULL, TRUE, FALSE, NULL );

	* iBufferSize = 0;

	if( osReader.hEvent == NULL )
	{
		/* Error creating overlapped event; abort. */
		return ERROR_SERIAL_OVERLAPPED_EVENT;
	}

	vSleep( 100 );

	do
	{	
		/* Issue read operation */
		dwAnswer = ReadFile( hlCom, ( pszBuffer + (*iBufferSize) ) , 1 , &dwRead, &osReader );

		dwAnswer = GetLastError();

		if( dwAnswer && dwRead == 0x00 )
		{
			CloseHandle( osReader.hEvent );

			// Log Serial Information
			#ifdef COMPILE_LOG_SERIAL
				clsLog.LogData( "LLI", pszBuffer, *iBufferSize, chAction );
			#endif
			// End of log Serial

			return SERIAL_NO_ERROR;
		}

		if( dwRead == 0x01 )
		{
			(*iBufferSize) += dwRead;
		}

	} while( !dwAnswer && ( * iBufferSize ) < MAX_BUFFER_LENGTH );
	
	CloseHandle( osReader.hEvent );

	// Log Serial Information
	#ifdef COMPILE_LOG_SERIAL
		clsLog.LogData( "LLI", pszBuffer, *iBufferSize, chAction );
	#endif
	// End of log Serial

	return SERIAL_NO_ERROR;
}
/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Parse_Answer( void )
{
	unsigned char		uchByteRead;
	int					iOperation_Answer;
	bool				bIntermediateAnswer;
	int					iMessagePosition;
	int					iState = STATE_WAIT_ACK;
	int					iCheckSumPosition;
	clock_t				clkStart, clkEnd;

	iMessagePosition = 0;
	szRecvMessage[iMessagePosition] = 0;
	//vSleep( 1000 );

	while( iState != STATE_MESSAGE_COMPLETE )
	{		
		clkStart = clock();

		/* Overlapped Timeout Correction */
		do
		{
  			iOperation_Answer = iCSerial_Receive_Buffer( &uchByteRead );
			clkEnd = clock();
		} while( iOperation_Answer != 0x00 && ( ( clkEnd - clkStart ) < FULL_TIMEOUT ) );

		if( ( ( clkEnd - clkStart ) > FULL_TIMEOUT ) && iOperation_Answer != 0 )
			return ERROR_SERIAL_TIMEOUT_ERROR;
																 
		switch( iState )
		{
			case STATE_WAIT_ACK:
				if( uchByteRead == STX_CHAR )
				{
					bIntermediateAnswer = false;
					iCheckSumPosition = 0;

					iState = STATE_WAIT_ETX;
					szRecvMessage[iMessagePosition] = uchByteRead;
					iMessagePosition++;
					break;
				}
				if( uchByteRead != ACK_CHAR )
				{
					if( uchByteRead == NCK_CHAR )
					{
						szRecvMessage[iMessagePosition] = uchByteRead;
						iMessagePosition++;

						// Log Serial Information
						#ifdef COMPILE_LOG_SERIAL
							clsLog.LogData( "LLI", szRecvMessage, iMessagePosition, chAction );
						#endif
						// End of log Serial

						return ERROR_SERIAL_NACK_ERROR;
					}
					else
						//return ERROR_SERIAL_TIMEOUT_ERROR; 
					{
						// wait, clean the buffer and return error
						vSleep( 2000 );
						do
						{ iOperation_Answer = iCSerial_Receive_Buffer( &uchByteRead );
						} while( iOperation_Answer == 0x00 );
						PurgeComm( hlCom, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);
						return ERROR_SERIAL_OVERLAPPED_EVENT; 
					}
				}
				
				iState = STATE_WAIT_STX;
				szRecvMessage[iMessagePosition] = uchByteRead;
				iMessagePosition++;
				break;
			case STATE_WAIT_STX:
				bIntermediateAnswer = false;
				iCheckSumPosition = 0;

				if( uchByteRead != STX_CHAR )
					return ERROR_SERIAL_TIMEOUT_ERROR; 

				iState = STATE_WAIT_ETX;
				szRecvMessage[iMessagePosition] = uchByteRead;
				iMessagePosition++;
				break;
			case STATE_WAIT_ETX:
				if( uchByteRead == ETX_CHAR )
				{
					int iEscCount;

					for( iEscCount = 0;
							 szRecvMessage[ iMessagePosition - ( 1 + iEscCount ) ] == 0x1b ;
							 iEscCount++ );

					if( !(iEscCount % 2) || iEscCount == 0 )
						iState = STATE_WAIT_CHECKSUM;
				}

				if( szRecvMessage[iMessagePosition - 1 ] == STX_CHAR && szRecvMessage[ iMessagePosition - 2] != ESC_CHAR )
					if(  uchByteRead == INT_CHAR )
						bIntermediateAnswer = true;
					else
					{
						iMessagePosition = 0;
					}
					
				szRecvMessage[iMessagePosition] = uchByteRead;
				iMessagePosition++;
				break;
			case STATE_WAIT_CHECKSUM:
				if( iCheckSumPosition == 3 )
				{
					if( bIntermediateAnswer == true )
					{
						// Log Serial Information
						#ifdef COMPILE_LOG_SERIAL
							clsLog.LogData( "LLI", szRecvMessage, iMessagePosition+1, chAction );
						#endif
						// End of log Serial

						iMessagePosition = 0;
						iState = STATE_WAIT_STX;
					}
					else
					{
						iState = STATE_MESSAGE_COMPLETE;
						szRecvMessage[iMessagePosition] = uchByteRead;
						iMessagePosition++;
					}

				}
				else
				{
					szRecvMessage[iMessagePosition] = uchByteRead;
					iMessagePosition++;
					iCheckSumPosition++;
				}
				break;
		}
	} 

	iRecvLength = iMessagePosition;
	szRecvMessage[ iMessagePosition ] = 0x00;

	// Log Serial Information
	#ifdef COMPILE_LOG_SERIAL
		clsLog.LogData( "LLI", szRecvMessage, iRecvLength, chAction );
	#endif
	// End of log Serial

	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Add_Field( unsigned char * pszField, size_t iFieldLength )
{
	int iAnswer;

	if( iSendLength + iFieldLength >= MAX_BUFFER_LENGTH )
		return ERROR_SERIAL_BUFFER_OVERFLOW;

	iAnswer = iCSerial_Add_FieldEx( pszField, iFieldLength, MESSAGE_TYPE_ASCII, true );
	iMessageType = SERIAL_MESSAGE_TYPE_1;
	return iAnswer;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Add_FieldEx( unsigned char *	szField      ,
								   size_t           iFieldLength ,
								   unsigned int		uiFieldType  ,
								   bool             bSeparator   )
{
	size_t iCounter = 0;
	int iDataFieldAux = iSendLength;

	if( iMaxDataAllowed + iFieldLength > MAX_SEND_LENGTH )
		return ERROR_SERIAL_BUFFER_OVERFLOW;

	iMaxDataAllowed += (int) iFieldLength;

	iMessageType = SERIAL_MESSAGE_TYPE_2;

	switch( uiFieldType )
	{
		case MESSAGE_TYPE_BINARY:
			unsigned char uchSoma;
			
			for( iCounter = 0; iCounter < iFieldLength; iCounter +=2 )
			{
				if( !bIs_Hexadecimal( szField[iCounter] ) || !bIs_Hexadecimal(szField[iCounter+1]) )
				{
					iSendLength = iDataFieldAux;
					return ERROR_SERIAL_INVALID_FIELD;
				}
				else
				{
					uchSoma = (unsigned char) sHexadecimal_Value(szField[iCounter], szField[iCounter + 1] );

					if( bIs_Escaped_Char( uchSoma ) )
						szSendMessage[ iSendLength++ ] = ESC_CHAR;

					szSendMessage[ iSendLength++ ] = uchSoma;
				}
			}

			if( bSeparator )
				szSendMessage[ iSendLength++ ] = SEP_CHAR;

			break;
		case MESSAGE_TYPE_ASCII:
			for( iCounter = 0; iCounter < iFieldLength; iCounter++ )
			{
				if( bIs_Escaped_Char( szField[ iCounter ] ) )
					szSendMessage[ iSendLength++ ] = ESC_CHAR;

				szSendMessage[ iSendLength++ ] = szField[ iCounter ]; 
			}

			if( bSeparator )
				szSendMessage[ iSendLength++ ] = SEP_CHAR;

			break;
		default:
			return ERROR_SERIAL_INVALID_FIELD_TYPE;
	}

	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
void CSerial::vCSerial_Purge_Message(void)
{
	iSendLength = 0;
	iMaxDataAllowed = 0;
	memset(szSendMessage,0,sizeof(szSendMessage) );
	iMessageType = SERIAL_MESSAGE_TYPE_1;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Get_Field( void *vAnswer, int *iAnswerSize, int iAnswerField )
{
	return iCSerial_Get_FieldEx( vAnswer, iAnswerSize, iAnswerField, ANSWER_TYPE_STRING );
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Get_FieldEx( void *vAnswer, int *iAnswerSize, int iAnswerField, int iAnswerType )
{
	int iCounter;

	if( iAnswerField <= 0 || iAnswerField > iCSerial_Get_Field_Counter() )
	{   
		*iAnswerSize = 0;
		return ERROR_SERIAL_INVALID_ANSWER_FIELD;
	}

	if( iAnswerField == ANSWER_FIELD_SEQUENCE_NUMBER && ( iAnswerType == ANSWER_TYPE_UNSIGNED_CHAR || iAnswerType == ANSWER_TYPE_STRING ))
	{
		unsigned char *uchAnswer = (unsigned char *) vAnswer;

		*iAnswerSize = 1;
		*uchAnswer = szRecvMessage[0];
		return SERIAL_NO_ERROR;
	}
	else
	{
		int iFieldCounter, iFieldPos;
		char szField[MAX_BUFFER_LENGTH];
		
		iFieldPos = 0;
		iFieldCounter = 1;

		memset( szField, 0, sizeof(szField) );

		for( iCounter = 1; iCounter < iRecvLength - 4; iCounter++ )
		{
			if( ( szRecvMessage[iCounter] == SEP_CHAR || szRecvMessage[iCounter] == ETX_CHAR ) &&
				szRecvMessage[iCounter - 1] != ESC_CHAR )
			{
				iFieldCounter++;
				
				if( iFieldCounter > iAnswerField )
					break;
				else
				{
					iFieldPos = 0;
					memset( szField, 0, sizeof(szField) );
				}
			}
			else
			{
				int iEscCount;

				for( iEscCount = 0; szRecvMessage[ iCounter - iEscCount ] == ESC_CHAR ; iEscCount++ );

				if( iEscCount % 2 )
					iCounter++;

				szField[iFieldPos] = szRecvMessage[iCounter];
				iFieldPos++;
			}
		}

		if( iFieldCounter < iAnswerField )
		{
			if( iAnswerType == ANSWER_TYPE_STRING )
			{
				unsigned char *pszAnswer = ( unsigned char *) vAnswer;
				pszAnswer[0] = NULL;
			}
			*iAnswerSize = 0;
			return ERROR_SERIAL_ANSWER_FIELD_NOT_FOUND;
		}

		switch( iAnswerType )
		{
			case ANSWER_TYPE_UNSIGNED_CHAR:
				{
					unsigned char *uchAnswer = (unsigned char *) vAnswer;

					*uchAnswer = szField[0];
					*iAnswerSize = 1;
					break;
				}
			case ANSWER_TYPE_CHAR:
				{
					char *uchAnswer = (char *) vAnswer;

					*uchAnswer = szField[0];
					*iAnswerSize = 1;
					break;
				}
			case ANSWER_TYPE_UNSIGNED_SHORT:
				{
					unsigned short *pusAnswer = ( unsigned short * ) vAnswer;

					if( iAnswerField >= ANSWER_FIELD_1 )
					{
						if( iField_Numeric_Value( (unsigned char *) szField, iFieldPos, (int *) pusAnswer) != SERIAL_NO_ERROR )
						{
							*pusAnswer = 0;
							*iAnswerSize = 0;
							return ERROR_SERIAL_INVALID_FIELD_TYPE;
						}
					}
					else
						*pusAnswer = MAKEUSHORT( szField[0], szField[1] );

					*iAnswerSize = 2;
					break;
				}
			case ANSWER_TYPE_SHORT:
				{
					short *psAnswer = ( short * ) vAnswer;

					if( iAnswerField >= ANSWER_FIELD_1 )
					{
						 if( iField_Numeric_Value( (unsigned char *) szField, iFieldPos, (int *) psAnswer) != SERIAL_NO_ERROR )
						 {
							 *psAnswer = 0;
							 *iAnswerSize = 0;
							 return ERROR_SERIAL_INVALID_FIELD_TYPE;
						 }
					}
					else
						*psAnswer = MAKESHORT( szField[0], szField[1] );

					*iAnswerSize = 2;
					break;
				}
			case ANSWER_TYPE_UNSIGNED_INT:
			case ANSWER_TYPE_UNSIGNED_LONG:
				{
					unsigned long *pulAnswer = (unsigned long *) vAnswer;

					if( iField_Numeric_Value( (unsigned char *) szField, iFieldPos, (int *) pulAnswer ) != SERIAL_NO_ERROR )
					{
						*pulAnswer = 0;
						*iAnswerSize = 0;
						return ERROR_SERIAL_INVALID_FIELD_TYPE;
					}
					*iAnswerSize = 4;
					break;
				}

			case ANSWER_TYPE_INT:
			case ANSWER_TYPE_LONG:
				{
					long *plAnswer = (long *) vAnswer;

					if( iField_Numeric_Value( (unsigned char *) szField, iFieldPos, (int *) plAnswer ) != SERIAL_NO_ERROR )
					{
						*plAnswer = 0;
						*iAnswerSize = 0;
						return ERROR_SERIAL_INVALID_FIELD_TYPE;
					}
					*iAnswerSize = 4;
					break;
				}

			case ANSWER_TYPE_STRING:
			case ANSWER_TYPE_BINARY:
				{
					unsigned char *pszAnswer = ( unsigned char *) vAnswer;
					int iCounter;

					for( iCounter = 0; iCounter < iFieldPos; iCounter++ )
					{
						pszAnswer[iCounter] = szField[iCounter];
					}
					pszAnswer[iCounter] = 0x00;
					
					*iAnswerSize = iFieldPos;

					break;
				}
			default:
				return ERROR_SERIAL_INVALID_FIELD_TYPE;
		}
	}

	return SERIAL_NO_ERROR;
}

/************************************************************************/
/* Project Name : DLL Padrao Afrac                                      */
/*	                                                                    */
/* Function     : CSerial::iCSerial_Calculate_CheckSum                  */
/*                                                                      */
/* Parameters   : None		                                            */
/*                                                                      */
/* Returns      : SERIAL_NO_ERROR OR ERROR_SERIAL_BUFFER_OVERFLOW       */
/************************************************************************/
/* Description  : This function adds the checksum at the end of the     */
/*                message.                                              */
/************************************************************************/
/* Author Name  : Marcus Vinicius Scolastici                            */
/************************************************************************/
/* Revision History                                                     */
/*                                                                      */
/* Date         Number    Revision Name     Author Name                 */
/* 29/05/2003    1.0      Initial Version   Marcus Vinicius Scolastici  */
/************************************************************************/
/* Revision Changes:                                                    */
/*                                                                      */
/* Number    Description                                                */
/*                                                                      */
/************************************************************************/
int CSerial::iCSerial_Get_Field_Counter( void )
{
	int iCount;
	int iFieldCounter = 0;
	int	iEscCounter = 0;

	for( iCount = 0; iCount < iRecvLength; iCount++ )	
	{
		if( iCount != 0 )
		{
			if( szRecvMessage[iCount - 1] == ESC_CHAR )
				iEscCounter++;
			else
				iEscCounter = 0;
		}


		if( ( szRecvMessage[iCount] == SEP_CHAR || szRecvMessage[iCount] == ETX_CHAR ) &&
			( ( szRecvMessage[iCount - 1] != ESC_CHAR ) ||
			  ( szRecvMessage[ iCount - 1] == ESC_CHAR &&  !( iEscCounter % 2 ) ) ) )
		{
			iFieldCounter++;
		}
	}

	return iFieldCounter;
}

void CSerial::vCSerial_Recover_Message( void )
{
	iSendLength = iAuxLength;
	iMaxDataAllowed = iMaxDataAllowedAux;
	memcpy( szSendMessage, szAuxMessage, iSendLength );
	iMessageType = iMsgAuxType;
}

bool CSerial::bCSerial_Is_Data_Available( void )
{
    DWORD		dwErrors;

    if( !ClearCommError( hlCom, &dwErrors, &comStat ) )
        return false;

    if( comStat.cbInQue )
		return true;

	return false;
}

int CSerial::iCSerial_Send_Buffer( unsigned char * szBuffer, int iBufferSize )
{
	// Log Serial Information
	#ifdef COMPILE_LOG_SERIAL
		clsLog.LogData( "LLO", szBuffer, iBufferSize, chAction );
	#endif
	// End of log Serial

	/* Purge Comm Port Data */
	PurgeComm( hlCom, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR );

	/* Copy Message */
	memcpy( szSendMessage, szBuffer, iBufferSize );
	iSendLength = iBufferSize;

	return iCSerial_Send_Buffer();
}

void CSerial::vCSerial_Receive_Buffer( unsigned char * szBuffer, int * iBufferSize )
{
	int		iPosition = 0;
			 
	*iBufferSize = 0x00;

	if( bCSerial_Is_Data_Available() == true )
	{
		iCSerial_Receive_Buffer( szBuffer, &iPosition );
		*iBufferSize += iPosition;
	}
	 
	PurgeComm( hlCom, PURGE_RXABORT | PURGE_TXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR );
}
