/*=============================================================================
** EFPBC.C                                                    EPSON Argentina
**=============================================================================
** EPSON Fiscal Printer Device driver example.
**=============================================================================
** Date             Author                  Comment
** 15-04-2004       Ricardo Malerba         Initial Version
**===========================================================================*/

/*=============================================================================
* Proposito: Muestra uso del driver DOS bajo un programa en "C"
* Operacion: Ticket de monto 1 + Cierre Z
* Compilador: BC/C++ 4.51
* Linea de comando: BCC -v -ml EFPBC.C
* Requisitos: Impresora fiscal serializada
*============================================================================*/

/* Include area */
#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <mem.h>
#include <string.h>
#include <fcntl.h>
#include <sys\stat.h>


/* Define Area */

/* Max cantidad de datos a transmitir */
#define EFP_MAXFIELDLEN        2048

/* Comandos del driver */
#define CMD_WRITE_ADD_FIELD      '0'
#define CMD_WRITE_SEND           '1'
#define CMD_WRITE_PURGE          '2'
#define CMD_READ_ANSWER          '3'
#define CMD_READ_EXTRA_FIELDS    '4'
#define CMD_READ_LOGS            '5'

/* Retornos */
#define EFP_Success               0
#define EFP_E_OutBufferFull      -1
#define EFP_E_InvalidDataInField -2
#define EFP_E_WriteFail          -3
#define EFP_E_ReadTimeOut        -4
#define EFP_E_NackReceived       -5
#define EFP_E_InvalidIndex       -6
#define EFP_E_BufferTooSmall     -7
#define EFP_E_InvalidCommand     -8

/* User types */

/* Tipos de datos no nativos */
typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned long dword;

typedef byte bool;
#define false 0
#define true (( bool ) !false )


/* Registro comn a todos los comandos */
typedef struct
{
  byte Cmd;              /* Tamao 1 byte  */
  word LastError;        /* Tamao 2 bytes */
}tHeader;

/* Registro para el comando "AddDataField" */
typedef struct
{
  tHeader Header;
  word Length;
  byte Buffer[ EFP_MAXFIELDLEN ];
}tAddDataField;

/* Registro para el comando "SendFrame" */
typedef struct
{
  tHeader Header;
}tSendFrame;

/* Registro para el comando "Purge" */
typedef struct
{
  tHeader Header;
}tPurge;

/* Regsitro para el comando "GetFinalAnswer" */
typedef struct
{
  tHeader Header;
  word FiscalStatus;
  word PrinterStatus;
  word ReturnCode;
  word ExtraFieldsCount;
}tFinalAnswer;

/* Registro para el comando "GetExtraField" */
typedef struct
{
  tHeader Header;
  word IndexLength;
  byte Buffer[ EFP_MAXFIELDLEN ];
}tExtraField;

/* Regsitro para el comando "GetLogs" */
typedef struct
{
  tHeader Header;
  word LogLen;
  byte LogBuffer[ 8192 ];
}tLogBuffer;

/* Variables globales */
int DriverHandle;
FILE *LogHandle;
byte LogBuffer[8193];
unsigned int _stklen = 0x4000;

/*-----------------------------------------------------------------------------
** Function Name : HandleError
** Description   : Maneja el error en operaciones del driver.
** Input         : Mensage y numero de error.
** Outupt        : Nada
** Notes         : 
-----------------------------------------------------------------------------*/
void HandleError( char * Message, int ErrorNum )
{
  printf( "Error : %s\n", Message );

  switch ( ErrorNum )
  {
    case EFP_E_OutBufferFull:    
      printf( "ErrorNum: Buffer de datos lleno.\n" );
      break;
    case EFP_E_InvalidDataInField:
      printf( "ErrorNum: Datos invalidos.\n" );
      break;
    case EFP_E_WriteFail:        
      printf( "ErrorNum: Operacion de escritura fallo.\n" );
      break;
    case EFP_E_ReadTimeOut:      
      printf( "ErrorNum : Error de comunicacion.\n");
      break;
    case EFP_E_NackReceived:     
      printf( "ErrorNum : Error de transferencia de datos.\n");
      break;
    case EFP_E_InvalidIndex:     
      printf( "ErrorNum : Indice invalido.\n" );
      break;
    case EFP_E_BufferTooSmall:   
      printf( "ErrorNum : Buffer es muy chico.\n" );
      break;
    case EFP_E_InvalidCommand:   
      printf( "ErrorNum : Comando invalido.\n" );
      break;
  }

  /* Termina el programa */
  exit( -1 );
} 
/*-----------------------------------------------------------------------------
** Function Name : AddField
** Description   : Suma datos a la trama de envio.
** Input         : Buffer y largo
** Outupt        : Nada
** Notes         : 
-----------------------------------------------------------------------------*/
void AddField( byte * Buffer, word Len )
{
  tAddDataField MyRecord;

  /* Tipo de comando para EFPHOST.SYS */
  MyRecord.Header.Cmd = CMD_WRITE_ADD_FIELD;

  /* Largo de los datos a enviar */
  MyRecord.Length = Len;

  /* Datos */
  memcpy( MyRecord.Buffer, Buffer, Len );

  /* Se enva el primer campo a EFPHOST.SYS. Unicamente se envia 
     la cantidad necesaria de bytes */
  write( DriverHandle, &MyRecord, sizeof( tHeader ) + 2 + Len );

  /* Se analiza "LastError" */
  if ( MyRecord.Header.LastError != EFP_Success )
  {
    /* Manejar el error */
    HandleError( "AddField - DriverError", MyRecord.Header.LastError );
  } 
}
/*-----------------------------------------------------------------------------
** Function Name : SendFrame
** Description   : Suma datos a la trama de envio.
** Input         : Nada
** Outupt        : Nada
** Notes         : 
-----------------------------------------------------------------------------*/
void SendFrame( void )
{
  tSendFrame MyRecord;

  /* Se enva todo a la impresora fiscal */
  MyRecord.Header.Cmd = CMD_WRITE_SEND;
  
  write( DriverHandle, &MyRecord, sizeof( tSendFrame ));

  /* Se analiza "LastError" */
  if ( MyRecord.Header.LastError != EFP_Success )
  {
    /* Manejar el error */
    HandleError( "SendFrame - DriverError", MyRecord.Header.LastError );
  } 
}
/*-----------------------------------------------------------------------------
** Function Name : GetFinalAnswer
** Description   : Obtiene la respuesta final de la impresora fiscal.
** Input         : FS,PS,RC y numeros de campos extras.
** Outupt        : True si el resultado fue Ok
** Notes         : 
-----------------------------------------------------------------------------*/
void GetFinalAnswer( word *FiscalStatus, word *PrinterStatus, word *ReturnCode,
                     word *ExtraFieldsCount )
{
  tFinalAnswer MyRecord;

  MyRecord.Header.Cmd = CMD_READ_ANSWER;

  read( DriverHandle, &MyRecord, sizeof( tFinalAnswer ) );

  if ( MyRecord.Header.LastError != EFP_Success )
  {
    /* Manejar el error */
    HandleError( "ReadFinalAnswer - DriverError", MyRecord.Header.LastError );
  } 

  *FiscalStatus = MyRecord.FiscalStatus;
  *PrinterStatus = MyRecord.PrinterStatus;
  *ReturnCode = MyRecord.ReturnCode;
  *ExtraFieldsCount = MyRecord.ExtraFieldsCount;
} 
/*-----------------------------------------------------------------------------
** Function Name : GetExtraField
** Description   : Obtiene un campo de informacio extra.
** Input         : FS,PS,RC y numeros de campos extras.
** Outupt        : True si el resultado fue Ok
** Notes         : 
-----------------------------------------------------------------------------*/
void GetExtraField( byte *Buffer, word *IndexLen )
{
  tExtraField MyRecord;

  MyRecord.Header.Cmd = CMD_READ_EXTRA_FIELDS;
  MyRecord.IndexLength = *IndexLen;

  read( DriverHandle, &MyRecord, sizeof( tExtraField ));

  if ( MyRecord.Header.LastError != EFP_Success )
  {
    /* Manejar el error */
    HandleError( "GetExtraField - DriverError", MyRecord.Header.LastError );
  } 

  /* El campo buscado se encuentra en MyRecord.ExtraField.Buffer y el
      largo del mismo en MyRecord.ExtraField.IndexLength */
  *IndexLen = MyRecord.IndexLength;
  
  memcpy( Buffer, MyRecord.Buffer, MyRecord.IndexLength );
}
/*-----------------------------------------------------------------------------
** Function Name : GetLog
** Description   : Obtiene el log de comunicaciones.
** Input         : Buffer del usuario y largo
** Outupt        : Nada
** Notes         : 
-----------------------------------------------------------------------------*/
void GetLog( byte *Buffer, word *BufferLen )
{
  tLogBuffer MyRecord;

  MyRecord.Header.Cmd = CMD_READ_LOGS;
  MyRecord.LogLen = *BufferLen;

  read( DriverHandle, &MyRecord, sizeof( tHeader ) + 2 + *BufferLen );

  if ( MyRecord.Header.LastError != EFP_Success )
  {
    /* Manejar el error */
    HandleError( "GetLog - DriverError", MyRecord.Header.LastError );
  }

  memcpy( Buffer, MyRecord.LogBuffer, MyRecord.LogLen );

  *BufferLen = MyRecord.LogLen;
}
/*-----------------------------------------------------------------------------
** Function Name : InitDriver
** Description   : Inicializa el driver.
** Input         : Nada
** Outupt        : Nada
** Notes         : 
-----------------------------------------------------------------------------*/
bool InitDriver( void )
{
  /* Abre EFPHOST.SYS */
  DriverHandle = open( "EFPHOST", O_RDWR | O_BINARY );

  if ( DriverHandle == -1 )
    return false;
  else
    return true;
} 
/*-----------------------------------------------------------------------------
** Function Name : TicketAndClosure
** Description   : Realiza un ticket y un cierre de jornada.
** Input         : Nada
** Outupt        : Nada
** Notes         : 
-----------------------------------------------------------------------------*/
void TicketAndClosure( void )
{
  byte Buffer[EFP_MAXFIELDLEN]; 
  word Index;
  word FS,PS,RC,GEFC;
  
  /* Fijar tipo de pago */
  printf( "Setea tipo de pago\n" );
  Buffer[ 0 ] = 0x05;
  Buffer[ 1 ] = 0x0C;
  AddField( Buffer, 2 );
  Buffer[ 0 ] = 0x00;
  Buffer[ 1 ] = 0x00;
  AddField( Buffer, 2 );
  AddField( "2", 1 );
  strcpy( Buffer, "Contado" );
  AddField( Buffer,  strlen( Buffer ));
  SendFrame();

  /* Actualiza LOG de comunicaciones */
  Index = sizeof( LogBuffer );
  GetLog( LogBuffer, &Index );
  LogBuffer[Index] = '\0';
  fwrite( LogBuffer, 1, Index, LogHandle );

  /* Muestra resultado del comando */
  GetFinalAnswer( &FS, &PS, &RC, &GEFC );
  printf( "FS: %04X PS: %04X RC: %04X\n", FS, PS, RC );

  /* Abrir boleta fiscal */
  printf( "\nAbre boleta fiscal\n" );
  Buffer[ 0 ] = 0x0A;
  Buffer[ 1 ] = 0x01;
  AddField( Buffer, 2 );
  Buffer[ 0 ] = 0x00;
  Buffer[ 1 ] = 0x00;
  AddField( Buffer, 2 );
  AddField( "", 0 );
  AddField( "", 0 );
  SendFrame();

  /* Actualiza LOG de comunicaciones */
  Index = sizeof( LogBuffer );
  GetLog( LogBuffer, &Index );
  LogBuffer[Index] = '\0';
  fwrite( LogBuffer, 1, Index, LogHandle );

  /* Muestra resultado del comando */
  GetFinalAnswer( &FS, &PS, &RC, &GEFC );
  printf( "FS: %04X PS: %04X RC: %04X\n", FS, PS, RC );

  /* Item */
  printf( "\nItem ... \n" );
  Buffer[ 0 ] = 0x0A;
  Buffer[ 1 ] = 0x02;
  AddField( Buffer, 2 );
  Buffer[ 0 ] = 0x00;
  Buffer[ 1 ] = 0x00;
  AddField( Buffer, 2 );
  strcpy( Buffer, "Descripcion Extra" );
  AddField( Buffer, strlen( Buffer ));
  AddField( Buffer, strlen( Buffer ));
  AddField( Buffer, strlen( Buffer ));
  AddField( Buffer, strlen( Buffer ));
  AddField( Buffer, strlen( Buffer ));
  strcpy( Buffer, "Descripcion del Item" );
  AddField( Buffer, strlen( Buffer ));
  AddField( "10000", 5 );
  AddField( "10000", 5 );
  AddField( "2100", 4 );
  SendFrame();

  /* Actualiza LOG de comunicaciones */
  Index = sizeof( LogBuffer );
  GetLog( LogBuffer, &Index );
  LogBuffer[Index] = '\0';
  fwrite( LogBuffer, 1, Index, LogHandle );

  /* Muestra resultado del comando */
  GetFinalAnswer( &FS, &PS, &RC, &GEFC );
  printf( "FS: %04X PS: %04X RC: %04X\n", FS, PS, RC );

  /* Pago */
  printf( "\nPago ...\n" );
  Buffer[ 0 ] = 0x0A;
  Buffer[ 1 ] = 0x05;
  AddField( Buffer, 2 );
  Buffer[ 0 ] = 0x00;
  Buffer[ 1 ] = 0x00;
  AddField( Buffer, 2 );
  AddField( "2", 1 );
  AddField( "1", 1 );
  SendFrame();
  
  /* Actualiza LOG de comunicaciones */
  Index = sizeof( LogBuffer );
  GetLog( LogBuffer, &Index );
  LogBuffer[Index] = '\0';
  fwrite( LogBuffer, 1, Index, LogHandle );

  /* Muestra resultado del comando */
  GetFinalAnswer( &FS, &PS, &RC, &GEFC );
  printf( "FS: %04X PS: %04X RC: %04X\n", FS, PS, RC );

  /* Cerrar */
  printf( "\nCierre ...\n" );
  Buffer[ 0 ] = 0x0A;
  Buffer[ 1 ] = 0x06;
  AddField( Buffer, 2 );
  Buffer[ 0 ] = 0x00;
  Buffer[ 1 ] = 0x01;
  AddField( Buffer, 2 );
  AddField( "1", 1 );
  strcpy( Buffer, "Gracias por su Compra" );
  AddField( Buffer, strlen( Buffer ));
  AddField( "", 0 );
  AddField( "", 0 );
  AddField( "", 0 );
  AddField( "", 0 );
  SendFrame();

  /* Actualiza LOG de comunicaciones */
  Index = sizeof( LogBuffer );
  GetLog( LogBuffer, &Index );
  LogBuffer[Index] = '\0';
  fwrite( LogBuffer, 1, Index, LogHandle );

  /* Muestra resultado del comando */
  GetFinalAnswer( &FS, &PS, &RC, &GEFC );
  printf( "FS: %04X PS: %04X RC: %04X\n", FS, PS, RC );

  /* Closure X */
  printf( "\nCierre Z\n" );
  Buffer[ 0 ] = 0x08;
  Buffer[ 1 ] = 0x01;
  AddField( Buffer, 2 );
  Buffer[ 0 ] = 0x00;
  Buffer[ 1 ] = 0x00;
  AddField( Buffer, 2 );
  SendFrame();
  
  /* Actualiza LOG de comunicaciones */
  Index = sizeof( LogBuffer );
  GetLog( LogBuffer, &Index );
  LogBuffer[Index] = '\0';
  fwrite( LogBuffer, 1, Index, LogHandle );

  /* Muestra resultado del comando */
  GetFinalAnswer( &FS, &PS, &RC, &GEFC );
  printf( "FS: %04X PS: %04X RC: %04X\n", FS, PS, RC );

  /* Muestra el numero de cierre Z */
  Index = 1;
  GetExtraField( Buffer, &Index );
  Buffer[Index] = '\0';
  printf( "Z numero %s\n", Buffer );
}
/*-----------------------------------------------------------------------------
** Function Name : Main
** Description   : Entrada principal del programa..
** Input         : Nada
** Outupt        : Nada
** Notes         : 
-----------------------------------------------------------------------------*/
void main( void )
{
  if ( InitDriver() == false )
  {
    printf( "Driver no presente !!!\n" );
    return;
  }

  /* Abre archivo de logeo */
  LogHandle = fopen( "EFPLOG.LOG", "a+b" );

  TicketAndClosure();

  /* Cierra archivo de logeo */
  fclose( LogHandle );
}

